<?php

namespace App\Http\Controllers\Accounts;

use App\Exports\ActivityCostExport;
use App\Exports\ActivityExport;
use App\Exports\BalanceExport;
use App\Exports\BudgetExport;
use App\Exports\ClearExport;
use App\Exports\CreditorExport;
use App\Exports\DebitorExport;
use App\Exports\IncomeExport;
use App\Exports\LedgerExport;
use App\Exports\PDCMExport;
use App\Exports\TrialExport;
use App\Http\Controllers\Controller;
use App\Models\AccountCategory;
use App\Models\AccountSubCategory;
use App\Models\BankDetailRegister;
use App\Models\ChartOfAccounts;
use App\Models\ChartOfAccountsOpening;
use App\Models\Transactions;
use App\Models\Bills;
use App\Models\CurrencyMaster;
use App\Models\Employees;
use App\Models\Party;
use App\Models\TransactionCharges;
use App\Models\BillCharges;
use App\Models\TransactionDetail;
use App\Models\TransactionMaster;
use App\Models\User;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
// use Livewire\Component;
use Barryvdh\DomPDF\Facade as PDF;
use Maatwebsite\Excel\Facades\Excel;

class AccountsReportController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }



    public function getAccountLedger($accountId, $startDate, $endDate)
    {
        // 1. Opening Balances
        $openingBalance = ChartOfAccountsOpening::where('chart_of_account_id', $accountId)
            ->select(
                DB::raw('SUM(debit_amount) as debit_opening'),
                DB::raw('SUM(credit_amount) as credit_opening')
            )
            ->first();

        // 2. Current Period Transactions
        $currentTransactions = TransactionDetail::where('chart_of_account_id', $accountId)
            ->whereHas('voucherMaster', function ($query) use ($startDate, $endDate) {
                $query->whereBetween('date', [$startDate, $endDate]);
            })
            ->select(
                DB::raw('SUM(debit) as debit_current'),
                DB::raw('SUM(credit) as credit_current')
            )
            ->first();

        // 3. Calculate Balances
        $debitBalance = ($openingBalance->debit_opening ?? 0) + ($currentTransactions->debit_current ?? 0);
        $creditBalance = ($openingBalance->credit_opening ?? 0) + ($currentTransactions->credit_current ?? 0);

        return [
            'debit_opening' => $openingBalance->debit_opening ?? 0,
            'credit_opening' => $openingBalance->credit_opening ?? 0,
            'debit_current' => $currentTransactions->debit_current ?? 0,
            'credit_current' => $currentTransactions->credit_current ?? 0,
            'debit_balance' => $debitBalance,
            'credit_balance' => $creditBalance,
        ];
    }

    public function indexAccountsLedger(Request $request)
    {
        $parent_accounts = ChartOfAccounts::whereIn('level', [2, 3])->get();
        $child_accounts = ChartOfAccounts::whereIn('level', [3, 4])->get();
        $currencies = CurrencyMaster::where('status', 1)->get();

        return view('accounts_report.account_ledger.index', compact('parent_accounts', 'child_accounts', 'currencies'));
    }

    public function reportAccountsLedger(Request $request)
    {
        try {
            $exclude_zero_balance = $request->has('exclude_zero_balance');
            $hide_cheque_date = $request->hide_cheque_date;
            $date_from = $request->date_from;
            $date_to = $request->date_to;
            $parent_account_id = $request->parent_account_id;
            $child_account = $request->child_account;
            $currency_id = $request->currency_id;
            $order_by = $request->order_by ?? 'account_code';

            // Fetch distinct chart_of_account_id values with account names
            $distinctAccounts = ChartOfAccounts::select('id', 'name')
                ->whereIn('level', [3, 4])
                ->when($parent_account_id, function ($query) use ($parent_account_id) {
                    // return $query->where('parent_account_id', $parent_account_id);
                    if (is_array($parent_account_id) && $parent_account_id[0] != null) {
                        return $query->whereIn('parent_account_id', $parent_account_id);
                    }
                })
                ->when($child_account, function ($query) use ($child_account) {
                    // dd($child_account);
                    if (is_array($child_account) && $child_account[0] != null) {
                        return $query->whereIn('id', $child_account);
                    }
                    // return $query->where('id', $child_account);
                })
                ->orderBy($order_by)
                ->get();

            $allLedgerReports = [];

            foreach ($distinctAccounts as $account) {
                // Fetch opening balance
                $openingBalance = ChartOfAccountsOpening::where('chart_of_account_id', $account->id)
                    ->value('debit_lc') ?? 0;
                // dd($account->id);
                // dd($openingBalance);


                // Fetch voucher details for the account, ensuring correct date ordering
                $voucherDetails = TransactionDetail::with('transaction')
                    ->where('transaction_detail.account_id', $account->id)
                    ->join('transaction_master', 'transaction_detail.transaction_master_id', '=', 'transaction_master.id')
                    ->where('transaction_master.status', 4)
                    ->whereNull('transaction_master.deleted_at')
                    ->whereNull('transaction_detail.deleted_at')
                    ->whereBetween('transaction_master.date', [$date_from, $date_to])
                    ->orderBy('transaction_master.date')
                    ->get();
                // dd($voucherDetails);

                $ledgerReport = [];
                $runningBalance = $openingBalance;

                foreach ($voucherDetails as $detail) {
                    // dd($detail->transaction);
                    // $debit = ($detail->debit_vc > 0) ? $detail->debit_vc : ($detail->debit_lc ?? 0);
                    // $credit = ($detail->credit_vc > 0) ? $detail->credit_vc : ($detail->credit_lc ?? 0);

                    if ($currency_id != 1) {
                        if (!empty($detail->transaction?->job_master_id)) {
                            if ($detail->transaction?->voucher_type_id == 10) {
                                $sum = TransactionCharges::where('transaction_master_id', $detail->transaction_master_id)
                                    ->where('currency', $currency_id)   // or ->where('currency_id', $currency_id)
                                    ->sum('net_amount_inc_tax');
                                $sum_local = TransactionCharges::where('transaction_master_id', $detail->transaction_master_id)
                                    ->where('currency', $currency_id)   // or ->where('currency_id', $currency_id)
                                    ->sum('local_amount');
                            } else {
                                $sum = BillCharges::where('transaction_master_id', $detail->transaction_master_id)
                                    ->where('currency', $currency_id)   // or ->where('currency_id', $currency_id)
                                    ->sum('net_amount_inc_tax');
                                $sum_local = BillCharges::where('transaction_master_id', $detail->transaction_master_id)
                                    ->where('currency', $currency_id)   // or ->where('currency_id', $currency_id)
                                    ->sum('local_amount');

                            }

                            if ($sum <= 0 && $sum_local <= 0) {
                                continue; // drop zero-sum job rows (intentional?)
                            }

                            $debit = (($detail->debit_vc ?? 0) > 0 || ($detail->debit_lc ?? 0) > 0) ? $sum : 0;
                            $debit_local = (($detail->debit_lc ?? 0) > 0) ? $sum_local : 0;
                            $credit = (($detail->credit_vc ?? 0) > 0 || ($detail->credit_lc ?? 0) > 0) ? $sum : 0;
                            $credit_local = (($detail->credit_lc ?? 0) > 0) ? $sum_local : 0;
                        } else {
                            // keep original values for non-job rows (you chose VC here)
                            $debit_local = $detail->debit_lc ?? 0;
                            $debit = $detail->debit_vc ?? 0;
                            $credit = $detail->credit_vc ?? 0;
                            $credit_local = $detail->credit_lc ?? 0;
                        }
                    } else {
                        $debit_local = $detail->debit_lc ?? 0;
                        $debit = $detail->debit_vc ?? 0;
                        $credit_local = $detail->credit_lc ?? 0;
                        $credit = $detail->credit_vc ?? 0;
                    }

                    // $debit = $detail->debit_lc ?? 0;
                    // $credit = $detail->credit_lc ?? 0;
                    if (($debit <= 0 && $credit <= 0) || ($debit_local <= 0 && $credit_local <= 0)) {
                        continue;
                    }
                    $runningBalance += $debit_local - $credit_local;


                    $ledgerReport[] = [
                        'date' => $detail->transaction?->date,
                        'voucher_no' => $detail->transaction?->voucher_no,
                        'transaction_id' => $detail->transaction_master_id,
                        'debit' => $debit_local,
                        'credit' => $credit_local,
                        'debit_vc' => $debit,
                        'credit_vc' => $credit,
                        'closing_balance' => $runningBalance,
                        'currency_id' => $detail->currency_id,
                        'exchange_rate' => $detail->exchange_rate ?? 1,
                        'narration' => $detail->narration,
                        'reference_number' => $detail->reference_number,
                        'reference_date' => $detail->reference_date,
                    ];
                }

                // Sum the debits and credits for the account
                // $debitTotal = $voucherDetails->sum('debit_lc');
                // $creditTotal = $voucherDetails->sum('credit_lc');
                $debitTotal = collect($ledgerReport)->sum('debit');
                $creditTotal = collect($ledgerReport)->sum('credit');
                // dd($ledgerReport);

                $totalBalance = $runningBalance;

                if (!$exclude_zero_balance || $totalBalance != 0) {
                    $allLedgerReports[$account->id] = [
                        'account_id' => $account->id,
                        'account_name' => $account->name,
                        'ledgerReport' => $ledgerReport,
                        'debit_opening' => $openingBalance,
                        'credit_opening' => 0, // Assuming you don't track credit opening balances
                        'debit_current' => $debitTotal,
                        'credit_current' => $creditTotal,
                        'debit_balance' => $runningBalance,
                        'credit_balance' => 0, // Or calculate if needed
                    ];
                }
            }

            // dd($allLedgerReports);

            $report_type = $request->report_type;
            $export_type = $request->export_type; // Check if export is requested

            if ($export_type) {
                $fileName = 'account_ledger_' . now()->format('Y-m-d') . '.' . ($export_type === 'csv' ? 'csv' : 'xlsx');
                return Excel::download(new LedgerExport($allLedgerReports, $report_type, $hide_cheque_date, $date_from, $date_to), $fileName);
            }

            return view('accounts_report.account_ledger.report', [
                'allLedgerReports' => $allLedgerReports,
                'report_type' => $report_type,
                'currency' => $currency_id,
                'hide_cheque_date' => $hide_cheque_date,
                'date_from' => $date_from,
                'date_to' => $date_to
            ]);
        } catch (\Exception $e) {
            dd($e);
            Log::error($e); // Log the error for debugging
            return redirect()->back()->with('error', 'Something went wrong. Please try again.');
        }
    }

    public function indexTrialBalance(Request $request)
    {
        $parent_accounts = ChartOfAccounts::whereNull('parent_account_id')->get();
        $child_accounts = ChartOfAccounts::whereNotNull('parent_account_id')->get();
        $currencies = CurrencyMaster::where('status', 1)->get();

        return view('accounts_report.trial_balance.index', compact('parent_accounts', 'child_accounts', 'currencies'));
    }

    public function reportTrialBalance(Request $request)
    {
        $date_from = $request->date_from ? addslashes($request->date_from) : null;
        $date_to = $request->date_to ? addslashes($request->date_to) : null;
        $currency_id = $request->currency_id ? (int) $request->currency_id : null;
        $parent_account_id = $request->parent_account_id;
        $child_account = $request->input('child_account', []);

        $order_by = $request->order_by ?? 'account_code';
        $report_type = $request->report_type;
        $exclude_zero_balance = $request->has('exclude_zero_balance');

        // Ensure that parent_account_id is treated as an array (even if it's a single value)
        $parentAccountIds = [];
        if ($parent_account_id) {
            $parentAccountIds = [$parent_account_id]; // Wrap in an array
        }

        // If child account is an array, use it directly, otherwise wrap single value in an array
        $childAccountIds = is_array($child_account) ? $child_account : [$child_account];

        // Get all account IDs by merging both parent and child family (ancestors + descendants)
        $accountIds = [];

        // If parent account is selected, get ancestors and descendants
        if (!empty($parentAccountIds)) {
            foreach ($parentAccountIds as $parentId) {
                $accountIds = array_merge(
                    $accountIds,
                    $this->getParentAccountIds($parentId), // Ancestors (grandparents, etc.)
                    $this->getChildAccountIds($parentId)   // Descendants (children, etc.)
                );
            }
        }

        // If child account is selected, get ancestors and descendants
        if (!empty($childAccountIds)) {
            foreach ($childAccountIds as $childId) {
                $accountIds = array_merge(
                    $accountIds,
                    $this->getParentAccountIds($childId),     // Ancestors (parents, grandparents, etc.)
                    $this->getChildAccountIds($childId)       // Descendants (children, grandchildren, etc.)
                );
            }
        }

        // Remove duplicates and prepare the final list of account IDs
        $accountIds = array_unique($accountIds);

        $trialBalance = DB::table('chart_of_accounts')->whereNull('deleted_at')
            // Filter by the account IDs
            ->when($accountIds, function ($query) use ($accountIds) {
                $query->whereIn('chart_of_accounts.id', $accountIds);
            })
            ->leftJoin(
                DB::raw('(SELECT chart_of_account_id, SUM(debit_amount) as opening_debit, SUM(credit_amount) as opening_credit
                     FROM chart_of_accounts_opening
                     GROUP BY chart_of_account_id) as opening_balance'),
                'chart_of_accounts.id',
                '=',
                'opening_balance.chart_of_account_id'
            )
            ->leftJoin(
                DB::raw('(
                    SELECT td.account_id,
                        SUM(td.debit_lc) as current_debit,
                        SUM(td.credit_lc) as current_credit
                    FROM transaction_detail td
                    INNER JOIN transaction_master tm
                        ON tm.id = td.transaction_master_id
                    WHERE tm.status = 4
                        AND tm.deleted_at IS NULL
                        AND td.deleted_at IS NULL' .
                    ($date_from ? " AND tm.date >= '" . $date_from . "'" : "") .
                    ($date_to ? " AND tm.date <= '" . $date_to . "'" : "") .
                    ($currency_id ? " AND tm.currency_id = '" . $currency_id . "'" : "") . '
                    GROUP BY td.account_id
                ) as transactions'),
                'chart_of_accounts.id',
                '=',
                'transactions.account_id'
            )
            ->select(
                'chart_of_accounts.id',
                'chart_of_accounts.account_code as account_id',
                'chart_of_accounts.name as account_name',
                'chart_of_accounts.parent_account_id',
                DB::raw('COALESCE(opening_balance.opening_debit, 0) as opening_debit'),
                DB::raw('COALESCE(opening_balance.opening_credit, 0) as opening_credit'),
                DB::raw('COALESCE(transactions.current_debit, 0) as current_debit'),
                DB::raw('COALESCE(transactions.current_credit, 0) as current_credit'),
                DB::raw('(COALESCE(opening_balance.opening_debit, 0) + COALESCE(transactions.current_debit, 0)) as balance_debit'),
                DB::raw('(COALESCE(opening_balance.opening_credit, 0) + COALESCE(transactions.current_credit, 0)) as balance_credit')
            )
            ->orderBy('chart_of_accounts.' . $order_by)
            ->groupBy('chart_of_accounts.id', 'chart_of_accounts.account_code', 'chart_of_accounts.name', 'chart_of_accounts.parent_account_id', 'opening_balance.opening_debit', 'opening_balance.opening_credit', 'transactions.current_debit', 'transactions.current_credit')
            ->get();

        // if ($exclude_zero_balance) {
        //     $trialBalance = $trialBalance->filter(function ($item) {
        //         return $item->balance_debit != 0 || $item->balance_credit != 0;
        //     });
        // }

        $export_type = $request->export_type; // Check if export is requested

        if ($export_type) {
            $fileName = 'trial_balance_' . now()->format('Y-m-d') . '.' . ($export_type === 'csv' ? 'csv' : 'xlsx');
            return Excel::download(new TrialExport($trialBalance, $report_type, $date_from, $date_to), $fileName);
        }

        return view('accounts_report.trial_balance.report', compact('trialBalance', 'report_type'));
    }

    // Get all parent account IDs (ancestors) for a given account ID
    private function getParentAccountIds($account_id)
    {
        $parentIds = [];
        if ($account_id) {
            $parentIds[] = $account_id;
            $parentAccount = DB::table('chart_of_accounts')->where('id', $account_id)->first();
            while ($parentAccount && $parentAccount->parent_account_id) {
                $parentIds[] = $parentAccount->parent_account_id;
                $parentAccount = DB::table('chart_of_accounts')->where('id', $parentAccount->parent_account_id)->first();
            }
        }
        return $parentIds;
    }

    // Get all child account IDs (descendants) for a given account ID
    private function getChildAccountIds($account_id)
    {
        $childIds = [];
        if ($account_id) {
            $childIds[] = $account_id;
            $childAccounts = DB::table('chart_of_accounts')->where('parent_account_id', $account_id)->get();
            foreach ($childAccounts as $childAccount) {
                $childIds = array_merge($childIds, $this->getChildAccountIds($childAccount->id)); // Recursive call for all descendants
            }
        }
        return $childIds;
    }


    public function indexIncomeStatement(Request $request)
    {
        $parent_accounts = ChartOfAccounts::whereIn('level', [2, 3])->where('category_id', 4)->get();
        $child_accounts = ChartOfAccounts::whereIn('level', [2, 3])->where('category_id', 3)->get();
        $currencies = CurrencyMaster::where('status', 1)->get();

        return view('accounts_report.income_statement.index', compact('parent_accounts', 'child_accounts', 'currencies'));
    }

    public function reportIncomeStatement(Request $request)
    {
        $startDate = $request->input('date_from');
        $endDate = $request->input('date_to');
        $parent_account_id = $request->input('parent_account_id');
        $child_account = $request->input('child_account');
        $currency_id = $request->input('currency_id');
        $exclude_zero_balance = $request->input('exclude_zero_balance', true);

        $accounts = ChartOfAccounts::whereIn('level', [3, 4])->get();

        $revenue = [];
        $expense = [];

        foreach ($accounts as $account) {
            // Opening balances filtered by currency
            $opening = $account->opening;

            $openingDebit = 0;
            $openingCredit = 0;

            if ($opening && (!$currency_id || $opening->currency_id == $currency_id)) {
                $openingDebit = $opening->debit_amount;
                $openingCredit = $opening->credit_amount;
            }

            // Current transactions filtered by currency and status
            $currentDebit = $account->transactionDetails()
                ->whereHas('transaction', function ($query) use ($currency_id, $startDate, $endDate) {
                    $query->where('status', 4);
                    if ($currency_id) {
                        $query->where('currency_id', $currency_id);
                    }

                    if ($startDate) {
                        $query->where('date', '>=', $startDate);
                    }

                    if ($endDate) {
                        $query->where('date', '<=', $endDate);
                    }
                })
                ->sum('debit_lc');

            $currentCredit = $account->transactionDetails()
                ->whereHas('transaction', function ($query) use ($currency_id, $startDate, $endDate) {
                    $query->where('status', 4);
                    if ($currency_id) {
                        $query->where('currency_id', $currency_id);
                    }

                    if ($startDate) {
                        $query->where('date', '>=', $startDate);
                    }

                    if ($endDate) {
                        $query->where('date', '<=', $endDate);
                    }
                })
                ->sum('credit_lc');

            $balanceDebit = $openingDebit + $currentDebit;
            $balanceCredit = $openingCredit + $currentCredit;

            if ($exclude_zero_balance && $balanceDebit == 0 && $balanceCredit == 0) {
                continue;
            }

            $data = [
                's_no' => $account->id,
                'code' => $account->account_code,
                'account_title' => $account->name,
                'opening_debit' => $openingDebit,
                'opening_credit' => $openingCredit,
                'current_debit' => $currentDebit,
                'current_credit' => $currentCredit,
                'balance_debit' => $balanceDebit,
                'balance_credit' => $balanceCredit,
            ];

            // Filter logic
            $includeRevenue = !$parent_account_id || ($account->pl_category_id == 4 && $account->id == $parent_account_id);
            $includeExpense = !$child_account || ($account->pl_category_id == 5 && $account->id == $child_account);

            if ($parent_account_id && $child_account) {
                // Clear both and include only matching records
                if ($includeRevenue) {
                    $revenue[] = $data;
                }
                if ($includeExpense) {
                    $expense[] = $data;
                }
            } elseif ($parent_account_id) {
                $expense = []; // Clear expense
                if ($includeRevenue) {
                    $revenue[] = $data;
                }
            } elseif ($child_account) {
                $revenue = []; // Clear revenue
                if ($includeExpense) {
                    $expense[] = $data;
                }
            } else {
                // Default include all
                if ($account->pl_category_id == 4) {
                    $revenue[] = $data;
                } elseif ($account->pl_category_id == 5) {
                    $expense[] = $data;
                }
            }
        }

        // Export or view
        $export_type = $request->export_type;

        if ($export_type) {
            $fileName = 'income_statement_' . now()->format('Y-m-d') . '.' . ($export_type === 'csv' ? 'csv' : 'xlsx');
            return Excel::download(new IncomeExport($revenue, $expense, $startDate, $endDate), $fileName);
        }

        return view('accounts_report.income_statement.report', compact('revenue', 'expense'));
    }


    private function updatePlCategories()
    {
        // Update revenue accounts
        ChartOfAccounts::where('account_code', 'like', '4%')->update(['pl_category_id' => 4]);

        // Update expense accounts
        ChartOfAccounts::where('account_code', 'like', '5%')->update(['pl_category_id' => 5]);
    }

    public function indexBalanceSheet(Request $request)
    {
        $parent_accounts = ChartOfAccounts::whereNull('parent_account_id')->get();
        $child_accounts = ChartOfAccounts::whereNotNull('parent_account_id')->get();
        $currencies = CurrencyMaster::where('status', 1)->get();
        $chart_of_accounts_level = ChartOfAccounts::select('level')->distinct()->pluck('level');

        return view('accounts_report.balance_sheet.index', compact('parent_accounts', 'child_accounts', 'currencies', 'chart_of_accounts_level'));
    }

    public function reportBalanceSheet(Request $request)
    {
        $date_to = $request->date_to;
        $parent_account_id = $request->parent_account_id;
        $account_level = $request->account_level;
        $child_account = $request->child_account;
        $currency_id = $request->currency_id;

        // Start with base query
        $chartQuery = ChartOfAccounts::query();

        if ($account_level) {
            $chartQuery->where('level', $account_level);
        } else {
            $chartQuery->whereIn('level', [3, 4]);
        }

        if ($parent_account_id) {
            $chartQuery->where('parent_account_id', $parent_account_id);
        }

        if ($child_account) {
            $chartQuery->where('id', $child_account);
        }

        $chartOfAccounts = $chartQuery->get();

        // Filter transactions by status, and apply date_to and currency_id
        $transactionDetails = TransactionDetail::whereHas('transaction', function ($query) use ($date_to, $currency_id) {
            $query->where('status', 4);

            if ($currency_id) {
                $query->where('currency_id', $currency_id);
            }

            if ($date_to) {
                $query->whereDate('date', '<=', $date_to);
            }
        })->get();

        $excludeZeroBalance = $request->has('exclude_zero_balance');

        $assets = $this->calculateCategory($chartOfAccounts, $transactionDetails, '1', $excludeZeroBalance);
        $liabilities = $this->calculateCategory($chartOfAccounts, $transactionDetails, '3', $excludeZeroBalance);
        $capital = $this->calculateCategory($chartOfAccounts, $transactionDetails, '2', $excludeZeroBalance);
        $revenue = $this->calculateCategory($chartOfAccounts, $transactionDetails, '4', $excludeZeroBalance);
        $expenses = $this->calculateCategory($chartOfAccounts, $transactionDetails, '5', $excludeZeroBalance);

        $export_type = $request->export_type; // Check if export is requested

        if ($export_type) {
            $fileName = 'balance_sheet_' . now()->format('Y-m-d') . '.' . ($export_type === 'csv' ? 'csv' : 'xlsx');
            return Excel::download(new BalanceExport($assets, $liabilities, $capital, $revenue, $expenses), $fileName);
        }

        return view('accounts_report.balance_sheet.report', compact('assets', 'liabilities', 'capital', 'revenue', 'expenses'));
    }

    private function calculateCategory($chartOfAccounts, $transactionDetails, $categoryCode, $excludeZeroBalance)
    {
        $categoryData = [];
        $total = 0;

        $accounts = $chartOfAccounts->filter(function ($account) use ($categoryCode) {
            return substr($account->account_code, 0, 1) == $categoryCode;
        });

        foreach ($accounts as $account) {
            $transactions = $transactionDetails->where('account_id', $account->id);
            $openingBalance = $account->opening ? ($account->opening->debit_amount ? $account->opening->debit_amount : $account->opening->credit_amount) : 0;
            $debit = $transactions->sum('debit_lc');
            $credit = $transactions->sum('credit_lc');

            if ($categoryCode == '1' || $categoryCode == '5') {
                $balance = $openingBalance + $debit - $credit;
            } else {
                $balance = $openingBalance + $credit - $debit;
            }

            if ($excludeZeroBalance && $balance == 0) {
                continue;
            }

            $category = $account->name;
            $subcategory = $account->alias;

            if (!isset($categoryData[$category])) {
                $categoryData[$category] = [];
            }

            $categoryData[$category][$subcategory] = $balance;
            $categoryData[$category]['total'] = ($categoryData[$category]['total'] ?? 0) + $balance;
            $total += $balance;
        }

        $categoryData['total'] = $total;

        return $categoryData;
    }
    public function indexAccountsActivity(Request $request)
    {
        $accounts = ChartOfAccounts::whereNotIn('level', [1, 2])->get();
        $users = User::all();
        $date_from = Carbon::now()->month >= 7
            ? Carbon::now()->year . '-07-01'
            : Carbon::now()->subYear()->year . '-07-01';
        $date_to = Carbon::now()->format('Y-m-d');


        return view('accounts_report.account_activity.index', compact('accounts', 'date_to', 'date_from', 'users'));
    }
    public function reportAccountsActivity(Request $request)
    {
        $dateFrom = $request->date_from;
        $dateTill = $request->date_to;
        $debit_account_id = $request->debit_account_id;
        $credit_account_id = $request->credit_account_id;
        $voucher_no = $request->voucher_no;
        $cheque_no = $request->cheque_no;
        $user_id = $request->user_id;
        $log_filter = $request->log_filter;
        $voucher_type = $request->voucher_type;
        $show_add_edit_log = $request->show_add_edit_log;
        $add_cost_center = $request->add_cost_center;
        $transaction_status = $request->transaction_status;


        $voucherDetails = TransactionDetail::with('transaction')
            ->join('transaction_master', 'transaction_detail.transaction_master_id', '=', 'transaction_master.id');

        if ($transaction_status == 'posted') {
            $voucherDetails->where('transaction_master.status', 4);
        } elseif ($transaction_status == 'unposted') {
            $voucherDetails->where('transaction_master.status', '!=', 4);
        }

        if ($dateFrom && $dateTill) {
            $voucherDetails->whereBetween('transaction_master.date', [$dateFrom, $dateTill]);
        }

        if ($debit_account_id) {
            $voucherDetails->where('transaction_detail.account_id', $debit_account_id);
        }

        if ($credit_account_id) {
            $voucherDetails->where('transaction_detail.account_id', $credit_account_id);
        }

        if ($voucher_no) {
            $voucherDetails->where('transaction_master.voucher_no', $voucher_no);
        }

        if ($cheque_no) {
            $voucherDetails->where('transaction_master.cheque_number', $cheque_no);
        }

        if ($voucher_type == 'bvp') {
            $voucherDetails->where('transaction_master.voucher_type_id', 1);
        }
        if ($voucher_type == 'brv') {
            $voucherDetails->where('transaction_master.voucher_type_id', 2);
        }
        if ($voucher_type == 'cn') {
            $voucherDetails->where('transaction_master.voucher_type_id', 8);
        }
        if ($voucher_type == 'cpv') {
            $voucherDetails->where('transaction_master.voucher_type_id', 3);
        }
        if ($voucher_type == 'crv') {
            $voucherDetails->where('transaction_master.voucher_type_id', 5);
        }
        if ($voucher_type == 'jv') {
            $voucherDetails->where('transaction_master.voucher_type_id', 6);
        }
        if ($voucher_type == 'pi') {
            $voucherDetails->where('transaction_master.voucher_type_id', 9);
        }
        if ($voucher_type == 'si') {
            $voucherDetails->where('transaction_master.voucher_type_id', 10);
        }
        if ($voucher_type == 'tv') {
            $voucherDetails->where('transaction_master.voucher_type_id', 7);
        }
        if ($voucher_type == 'dn') {
            $voucherDetails->where('transaction_master.voucher_type_id', 4);
        }

        if ($show_add_edit_log == 1) {
            if ($user_id) {
                if ($log_filter == 'created_by') {
                    $voucherDetails->where('transaction_master.created_by', $user_id);
                } else if ($log_filter == 'edited_by') {
                    $voucherDetails->where('transaction_master.updated_by', $user_id);
                } else {
                    $voucherDetails->where(function ($query) use ($user_id) {
                        $query->where('transaction_master.created_by', $user_id)
                            ->orWhere('transaction_master.updated_by', $user_id);
                    });
                }
            }
        }

        $voucherDetails = $voucherDetails->get();

        $export_type = $request->export_type; // Check if export is requested

        if ($export_type) {
            $fileName = 'account_activity_' . now()->format('Y-m-d') . '.' . ($export_type === 'csv' ? 'csv' : 'xlsx');
            return Excel::download(new ActivityExport($voucherDetails, $add_cost_center), $fileName);
        }
        // dd($voucherDetails);
        return view('accounts_report.account_activity.report', compact('voucherDetails', 'add_cost_center'));
    }
    // public function reportAccountsActivityCostCenterWise(Request $request)
    // {
    //     $dateFrom = $request->date_from;
    //     $dateTill = $request->date_to;

    //     // Fetch data with necessary joins
    //     $voucherDetails = DB::table('transaction_detail')
    //         ->join('transaction_master', 'transaction_detail.transaction_master_id', '=', 'transaction_master.id')
    //         ->join('chart_of_accounts as child_account', 'transaction_detail.account_id', '=', 'child_account.id')
    //         ->leftJoin('chart_of_accounts as parent_account', 'child_account.parent_account_id', '=', 'parent_account.id') // Self-join for parent account name
    //         ->join('accounts_category_master', 'child_account.category_id', '=', 'accounts_category_master.id')
    //         ->select(
    //             'accounts_category_master.name as category_name', // Category name
    //             'parent_account.name as parent_account_name', // Parent account name
    //             'child_account.name as account_name', // Child account name
    //             'transaction_detail.debit_lc', // Debit value
    //             'transaction_master.date' // Transaction date
    //         )
    //         ->whereNull('transaction_detail.deleted_at') // Filter out soft-deleted records
    //         ->whereNull('transaction_master.deleted_at');

    //     // Apply date range filter if provided
    //     if ($dateFrom && $dateTill) {
    //         $voucherDetails->whereBetween('transaction_master.date', [$dateFrom, $dateTill]);
    //     }

    //     // Get the data and group by category name, then by parent account name
    //     $voucherDetails = $voucherDetails->get()
    //         ->groupBy('category_name') // Group by category
    //         ->map(function ($category) {
    //             return $category->groupBy('parent_account_name'); // Then group by parent account name within each category
    //         });

    //     // dd($voucherDetails);

    //     // Pass the grouped data to the view
    //     return view('accounts_report.account_activity_costcenter_wise.report', compact('voucherDetails'));
    // }
    public function reportAccountsActivityCostCenterWise(Request $request)
    {
        $dateFrom = $request->date_from;
        $dateTill = $request->date_to;
        $account_id = $request->account_id;
        // dd($account_id);

        $selectedCategories = [];
        if ($request->has('assets')) {
            $selectedCategories[] = 'Assets';
        }
        if ($request->has('capital_expense')) {
            $selectedCategories[] = 'Capital';
        }
        if ($request->has('expense')) {
            $selectedCategories[] = 'Expense';
        }
        if ($request->has('liabilities')) {
            $selectedCategories[] = 'Liability';
        }
        if ($request->has('revenue')) {
            $selectedCategories[] = 'Income';
        }

        // Fetch data with necessary joins
        $voucherDetails = DB::table('transaction_detail')
            ->join('transaction_master', 'transaction_detail.transaction_master_id', '=', 'transaction_master.id')
            ->join('chart_of_accounts as child_account', 'transaction_detail.account_id', '=', 'child_account.id')
            ->leftJoin('chart_of_accounts as parent_account', 'child_account.parent_account_id', '=', 'parent_account.id') // Self-join for parent account name
            ->join('accounts_category_master', 'child_account.category_id', '=', 'accounts_category_master.id')
            ->select(
                'accounts_category_master.name as category_name', // Category name
                'parent_account.name as parent_account_name', // Parent account name
                'child_account.name as account_name', // Child account name
                'transaction_detail.debit_lc', // Debit value
                'transaction_detail.credit_lc', // Credit value
                'transaction_master.date' // Transaction date
            )
            ->whereNull('transaction_detail.deleted_at') // Filter out soft-deleted records
            ->whereNull('transaction_master.deleted_at');

        if ($dateFrom && $dateTill) {
            $voucherDetails->whereBetween('transaction_master.date', [$dateFrom, $dateTill]);
        }

        if (!empty($selectedCategories)) {
            $voucherDetails->whereIn('accounts_category_master.name', $selectedCategories);
        }

        // Filter by account_id
        // if ($account_id) {
        //     $voucherDetails->where('transaction_detail.account_id', $account_id);
        // }
        if (!empty($account_id)) {
            $voucherDetails->where(function ($query) use ($account_id) {
                $query->where('transaction_detail.account_id', $account_id)
                    ->orWhere('parent_account.id', $account_id);
            });
        }

        // Get the data and group by category name, then by parent account name
        $voucherDetails = $voucherDetails->get()
            ->groupBy('category_name') // Group by category
            ->map(function ($category) {
                return $category->groupBy('parent_account_name'); // Then group by parent account name within each category
            });
        // dd($voucherDetails);

        return view('accounts_report.account_activity_costcenter_wise.report', compact('voucherDetails'));
    }

    public function reportPDCM(Request $request)
    {
        // dd($request->all());
        $up_till_cheque_date = $request->up_till_cheque_date;
        $date = $request->date;
        $show_unposted_record = $request->show_unposted_record;
        $bank_id = $request->bank_id;
        $client_id = $request->client_id;
        $report_type = $request->report_type;

        $voucherDetails = TransactionDetail::with('transaction')
            ->join('transaction_master', 'transaction_detail.transaction_master_id', '=', 'transaction_master.id')
            ->join('chart_of_accounts as child', 'transaction_detail.account_id', '=', 'child.id')
            ->join('chart_of_accounts as parent', 'child.parent_account_id', '=', 'parent.id')
            ->where('child.sub_category_id', 2)
            ->whereIn('transaction_master.transaction_type_master_id', [1, 2])
            ->select(
                'parent.id as parent_account_id',
                'parent.name as parent_account_name',
                'transaction_master.*',
                'transaction_detail.*',
                'child.*'
            );

        if ($up_till_cheque_date == 1) {
            $voucherDetails = $voucherDetails->whereDate('transaction_master.cheque_date', '<=', $date);
        }

        if ($show_unposted_record == 1) {
            $voucherDetails->where('transaction_master.status', '!=', 4);
        } else {
            $voucherDetails->where('transaction_master.status', 4);
        }

        if ($bank_id) {
            $voucherDetails->where('transaction_detail.account_id', $bank_id);
        }

        if ($client_id) {
            $voucherDetails->where('transaction_master.party_id', $client_id);
        }

        $voucherDetails = $voucherDetails->get()->groupBy('name');

        $voucherDetails = $voucherDetails->map(function ($group) {
            return $group->sortBy('name');
        });

        $export_type = $request->export_type; // Check if export is requested

        if ($export_type) {
            $fileName = 'pdcm_' . now()->format('Y-m-d') . '.' . ($export_type === 'csv' ? 'csv' : 'xlsx');
            return Excel::download(new PDCMExport($voucherDetails, $report_type), $fileName);
        }

        // dd($voucherDetails);

        return view('accounts_report.pdcm.report', compact('voucherDetails', 'report_type'));
    }
    public function reportCreditor_list(Request $request)
    {
        $dateFrom = $request->date_from;
        $dateTill = $request->date_to;
        $account_id = $request->account_id;
        $currency_id = $request->currency_id;
        $order_by = $request->order_by;

        $voucherDetails = TransactionDetail::with('transaction')
            ->join('transaction_master', 'transaction_detail.transaction_master_id', '=', 'transaction_master.id')
            ->join('chart_of_accounts as child', 'transaction_detail.account_id', '=', 'child.id')
            ->join('chart_of_accounts as parent', 'child.parent_account_id', '=', 'parent.id')
            ->where('child.sub_category_id', 4)
            // ->whereIn('child.sub_category_id', [3, 4])
            ->select(
                'parent.id as parent_account_id',
                'parent.name as parent_account_name',
                'transaction_master.*',
                'transaction_detail.*',
                'child.*'
            );

        if ($dateFrom && $dateTill) {
            $voucherDetails->whereBetween('transaction_master.date', [$dateFrom, $dateTill]);
        }

        if (!empty($currency_id)) {
            $voucherDetails->where('transaction_master.currency_id', $currency_id);
        }

        if (!empty($account_id)) {
            $voucherDetails->where(function ($query) use ($account_id) {
                $query->where('transaction_detail.account_id', $account_id)
                    ->orWhere('parent.id', $account_id);
            });
        }

        $voucherDetails = $voucherDetails->get()->groupBy('parent_account_id');

        if ($order_by == 'account_code') {
            $voucherDetails = $voucherDetails->map(function ($group) {
                return $group->sortBy('account_code');
            });
        } else {
            $voucherDetails = $voucherDetails->map(function ($group) {
                return $group->sortBy('name');
            });
        }

        $export_type = $request->export_type; // Check if export is requested

        if ($export_type) {
            $fileName = 'creditor_list_' . now()->format('Y-m-d') . '.' . ($export_type === 'csv' ? 'csv' : 'xlsx');
            return Excel::download(new CreditorExport($voucherDetails), $fileName);
        }
        return view('accounts_report.creditor_list.report', compact('voucherDetails'));
    }
    public function reportdebitor_list(Request $request)
    {
        $dateFrom = $request->date_from;
        $dateTill = $request->date_to;
        $account_id = $request->account_id;
        $currency_id = $request->currency_id;
        $order_by = $request->order_by;

        $voucherDetails = TransactionDetail::with('transaction')
            ->join('transaction_master', 'transaction_detail.transaction_master_id', '=', 'transaction_master.id')
            ->join('chart_of_accounts as child', 'transaction_detail.account_id', '=', 'child.id')
            ->join('chart_of_accounts as parent', 'child.parent_account_id', '=', 'parent.id')
            ->where('child.sub_category_id', 3)
            // ->whereIn('child.sub_category_id', [3, 4])
            ->select(
                'parent.id as parent_account_id',
                'parent.name as parent_account_name',
                'transaction_master.*',
                'transaction_detail.*',
                'child.*'
            );

        if ($dateFrom && $dateTill) {
            $voucherDetails->whereBetween('transaction_master.date', [$dateFrom, $dateTill]);
        }

        if (!empty($currency_id)) {
            $voucherDetails->where('transaction_master.currency_id', $currency_id);
        }

        if (!empty($account_id)) {
            $voucherDetails->where(function ($query) use ($account_id) {
                $query->where('transaction_detail.account_id', $account_id)
                    ->orWhere('parent.id', $account_id);
            });
        }



        $voucherDetails = $voucherDetails->get()->groupBy('parent_account_id');

        if ($order_by == 'account_code') {
            $voucherDetails = $voucherDetails->map(function ($group) {
                return $group->sortBy('account_code');
            });
        } else {
            $voucherDetails = $voucherDetails->map(function ($group) {
                return $group->sortBy('name');
            });
        }


        $export_type = $request->export_type; // Check if export is requested

        if ($export_type) {
            $fileName = 'debitor_list_' . now()->format('Y-m-d') . '.' . ($export_type === 'csv' ? 'csv' : 'xlsx');
            return Excel::download(new DebitorExport($voucherDetails), $fileName);
        }


        return view('accounts_report.debitor_list.report', compact('voucherDetails'));
    }
    public function reportchartofaccountlist(Request $request)
    {
        $account = $request->account;
        $account_id = $request->parent_account_id;
        $category_id = $request->category_id;
        $sub_category_id = $request->sub_category_id;
        $order_by = $request->order_by;

        $query = ChartOfAccounts::whereIn('level', [1, 2, 3, 4])->whereNotNull('account_code');

        if ($account == 'particular_parent') {
            $query->where('parent_account_id', $account_id);
        }

        if ($category_id) {
            $query->where('category_id', $category_id);
        }

        if ($sub_category_id) {
            $query->where('sub_category_id', $sub_category_id);
        }

        // If the order is based on hierarchical structure
        if ($order_by == 'account_code') {
            $accounts = $query->with([
                'account_category',
                'account_sub_category',
                'parent_account'
            ])->get();

            // Ensure the hierarchical sorting is applied
            $accounts = $this->sortHierarchically($accounts);
        } else {
            $accounts = $query->with([
                'account_category',
                'account_sub_category',
                'parent_account'
            ])->orderBy('name')->get();
        }

        return view('accounts_report.chart_of_account_list.report', compact('accounts'));
    }

    /**
     * Sort accounts hierarchically.
     *
     * @param \Illuminate\Support\Collection $accounts
     * @return \Illuminate\Support\Collection
     */
    protected function sortHierarchically($accounts)
    {
        $sorted = collect();

        // Build a map of accounts by parent_account_id
        $accountsByParent = $accounts->groupBy('parent_account_id');

        // Recursive function to add accounts in a hierarchical order
        $addAccounts = function ($parentId) use (&$addAccounts, &$sorted, $accountsByParent) {
            if (isset($accountsByParent[$parentId])) {
                foreach ($accountsByParent[$parentId] as $account) {
                    $sorted->push($account);
                    $addAccounts($account->id); // Add child accounts
                }
            }
        };

        // Start with root accounts (parent_account_id = null)
        $addAccounts(null);

        return $sorted;
    }
    public function reportCashbook(Request $request)
    {
        // Get the selected accounts
        $accounts = $request->input('account_ids', []);

        // Start and end dates for filtering transactions
        $startDate = $request->input('date_from');
        $endDate = $request->input('date_to');

        // Get the data for the report based on selected accounts and date range
        $transactions = $this->getCashBookTransactions($accounts, $startDate, $endDate);
        // dd($transactions);

        // Pass number of accounts to view for dynamic column rendering
        return view('accounts_report.cash_book.report', compact('transactions', 'accounts'));
    }
    /**
     * Fetch cash book transactions for the selected accounts
     */
    private function getCashBookTransactions(array $accounts, $startDate, $endDate)
    {
        $transactions = [];

        // Loop through each selected account to fetch its transactions
        foreach ($accounts as $accountId) {
            // Query to get the transactions for this account
            $accountTransactions = DB::table('transaction_detail as td')
                ->join('transaction_master as tm', 'td.transaction_master_id', '=', 'tm.id')
                ->join('chart_of_accounts as coa', 'td.account_id', '=', 'coa.id')
                ->where('td.account_id', $accountId)
                ->whereBetween('tm.created_at', [$startDate, $endDate])
                ->select(
                    'tm.voucher_no',
                    'tm.cheque_number',
                    'tm.date as transaction_date',
                    'coa.account_code',
                    'coa.name as account_title',
                    'td.narration', // Assuming 'particulars' holds the narration
                    'coa.name as account_name', // Account Name (to show in dynamic columns)
                    DB::raw('CASE WHEN td.debit_lc IS NULL OR td.debit_lc = 0 THEN td.credit_lc ELSE td.debit_lc END AS amount') // Handle debit/credit logic
                )
                ->get();

            // Add the transactions for this account to the result array
            $transactions[$accountId] = $accountTransactions;
        }

        return $transactions;
    }
    public function reportBudgetVariance(Request $request)
    {
        $dateFrom = $request->date_from;
        $dateTill = $request->date_to;
        $account_id = $request->account_id;
        $currency_id = $request->currency_id;

        $voucherDetails = TransactionDetail::with('transaction')
            ->join('transaction_master', 'transaction_detail.transaction_master_id', '=', 'transaction_master.id')
            ->join('chart_of_accounts as child', 'transaction_detail.account_id', '=', 'child.id')
            ->join('chart_of_accounts as parent', 'child.parent_account_id', '=', 'parent.id')
            // ->where('child.sub_category_id', 3)
            ->select(
                'child.name as account_name',
                'transaction_master.voucher_no',
                'transaction_master.date',
                'transaction_master.narration',
                'transaction_detail.debit_lc',
                'transaction_detail.credit_lc',
            );

        if ($dateFrom && $dateTill) {
            $voucherDetails->whereBetween('transaction_master.date', [$dateFrom, $dateTill]);
        }

        if (!empty($currency_id)) {
            $voucherDetails->where('transaction_master.currency_id', $currency_id);
        }

        if (!empty($account_id)) {
            $voucherDetails->where(function ($query) use ($account_id) {
                $query->where('transaction_detail.account_id', $account_id)
                    ->orWhere('parent.id', $account_id);
            });
        }

        // Retrieve data and format it
        $voucherDetails = $voucherDetails->get()->groupBy(function ($item) {
            return $item->account_name . ' - HEAD OFFICE - ' . Carbon::parse($item->date)->format('M, Y');
        });

        // Format for output
        $formattedDetails = [];
        foreach ($voucherDetails as $groupKey => $transactions) {
            $totalDebit = $transactions->sum('debit_lc');
            $totalCredit = $transactions->sum('credit_lc');
            $balance = $totalDebit - $totalCredit;
            $balances = 0;

            $groupData = [
                'group_key' => $groupKey,
                'transactions' => $transactions->map(function ($item) use (&$balances) {
                    $balances += $item->debit_lc;
                    $balances -= $item->credit_lc;
                    return [
                        'voucher_no' => $item->voucher_no,
                        'date' => Carbon::parse($item->date)->format('d/m/Y'),
                        'particulars' => $item->narration,
                        'debit' => $item->debit_lc,
                        'credit' => $item->credit_lc,
                        'balance' => abs($balances),
                        'budget_balance' => $balances,
                    ];
                })->toArray(),
                'totals' => [
                    'total_debit' => $totalDebit,
                    'total_credit' => $totalCredit,
                    'balance' => abs($balance),
                    'budget_balance' => $balance,
                ],
            ];

            $formattedDetails[] = $groupData;
        }

        $export_type = $request->export_type; // Check if export is requested

        if ($export_type) {
            $fileName = 'budget_variance_' . now()->format('Y-m-d') . '.' . ($export_type === 'csv' ? 'csv' : 'xlsx');
            return Excel::download(new BudgetExport($formattedDetails), $fileName);
        }
        // dd($formattedDetails);
        return view('accounts_report.budget_variance_report.report', compact('formattedDetails'));
    }
    public function reportAgeingDetail(Request $request)
    {
        $dateFrom = $request->date_from;
        $dateTill = $request->date_to;
        $account_id = $request->account_id;
        $currency_id = $request->currency_id;

        // Get transactions for invoices (Receivables)
        $receivables = Transactions::join('transaction_master', 'transactions.id', '=', 'transaction_master.transaction_id')
            ->join('currency_master', 'transaction_master.currency_id', '=', 'currency_master.id')
            ->join('transaction_detail', 'transaction_master.id', '=', 'transaction_detail.transaction_master_id')
            ->join('chart_of_accounts as coa', 'transaction_detail.account_id', '=', 'coa.id')
            ->where('transaction_detail.debit_lc', '>', 0)
            ->when($dateFrom, function ($query) use ($dateFrom) {
                return $query->whereDate('transactions.inv_date', '>=', $dateFrom);
            })
            ->when($dateTill, function ($query) use ($dateTill) {
                return $query->whereDate('transactions.inv_date', '<=', $dateTill);
            })
            ->when($account_id, function ($query) use ($account_id) {
                return $query->where('coa.id', $account_id);
            })
            ->when($currency_id, function ($query) use ($currency_id) {
                return $query->where('transaction_master.currency_id', $currency_id);
            })
            ->select([
                'transactions.inv_date as Date',
                'transactions.due_days as DueDate',
                'transaction_master.voucher_no as VoucherNo',
                'coa.account_code as Code',
                'transactions.tran_no as InvoiceNo',
                'transaction_detail.narration as Particular',
                'currency_master.name as curr',
                'transaction_detail.debit_lc as Receivable',
                'transaction_detail.credit_lc as Paid',
                DB::raw('(transaction_detail.debit_lc - transaction_detail.credit_lc) as Balance')
            ])
            ->get();

        // Get bills (Payables)
        $payables = Bills::join('transaction_master', 'bills.id', '=', 'transaction_master.bill_id')
            ->join('currency_master', 'transaction_master.currency_id', '=', 'currency_master.id')
            ->join('transaction_detail', 'transaction_master.id', '=', 'transaction_detail.transaction_master_id')
            ->join('chart_of_accounts as coa', 'transaction_detail.account_id', '=', 'coa.id')
            ->where('transaction_detail.credit_lc', '>', 0)
            ->when($dateFrom, function ($query) use ($dateFrom) {
                return $query->whereDate('bills.bill_date', '>=', $dateFrom);
            })
            ->when($dateTill, function ($query) use ($dateTill) {
                return $query->whereDate('bills.bill_date', '<=', $dateTill);
            })
            ->when($account_id, function ($query) use ($account_id) {
                return $query->where('coa.id', $account_id);
            })
            ->when($currency_id, function ($query) use ($currency_id) {
                return $query->where('transaction_master.currency_id', $currency_id);
            })
            ->select([
                'bills.bill_date as Date',
                'bills.due_days as DueDate',
                'transaction_master.voucher_no as VoucherNo',
                'coa.account_code as Code',
                'bills.tran_no as InvoiceNo',
                'transaction_detail.narration as Particular',
                'currency_master.name as curr',
                'transaction_detail.credit_lc as Payable',
                'transaction_detail.debit_lc as Received',
                DB::raw('(transaction_detail.credit_lc - transaction_detail.debit_lc) as Balance')
            ])
            ->get();

        // Pass data to the view
        return view('accounts_report.ageing_report.report', compact('receivables', 'payables'));
    }

    public function reportChequePrinting(Request $request)
    {
        $dateFrom = $request->date_from;
        $dateTill = $request->date_to;
        $id = $request->voucher_no;
        $transaction_master = TransactionMaster::where('id', $id)
            ->when($dateFrom, function ($query) use ($dateFrom) {
                return $query->whereDate('date', '>=', $dateFrom);
            })
            ->when($dateTill, function ($query) use ($dateTill) {
                return $query->whereDate('date', '<=', $dateTill);
            })->first();

        return view('accounts_report.cheque_printing.report', compact('transaction_master'));
    }
    public function reportInvoiceSettelment(Request $request)
    {
        $dateFrom = $request->date_from;
        $dateTill = $request->date_to;
        $account_id = $request->account_id;
        $currency_id = $request->currency_id;

        $invoices = TransactionMaster::with('detail', 'chart_of_account', 'chart_of_accounts', 'transaction', 'bill');

        if ($account_id) {
            $invoices->where(function ($query) use ($account_id) {
                $query->where('account_id', $account_id)
                    ->orWhere('party_id', $account_id);
            });
        }

        if ($currency_id) {
            $invoices->where('currency_id', $currency_id);
        }

        $invoices = $invoices->get();
        // dd($invoices);
        return view('accounts_report.invoice_settelment.report', compact('invoices'));
    }
    public function indexAccountsActivityCostCenterWise(Request $request)
    {
        $accounts = ChartOfAccounts::whereNotIn('level', [1, 2])->get();
        $date_from = Carbon::now()->month >= 7
            ? Carbon::now()->year . '-07-01'
            : Carbon::now()->subYear()->year . '-07-01';
        $date_to = Carbon::now()->format('Y-m-d');


        return view('accounts_report.account_activity_costcenter_wise.index', compact('accounts', 'date_from', 'date_to'));
    }
    public function indexDebitorList(Request $request)
    {
        $accounts = ChartOfAccounts::whereNotIn('level', [1, 2])->get();
        $date_from = Carbon::now()->month >= 7.
            ? Carbon::now()->year . '-07-01'
            : Carbon::now()->subYear()->year . '-07-01';
        $date_to = Carbon::now()->format('Y-m-d');
        $currencies = CurrencyMaster::where('status', 1)->get();

        return view('accounts_report.debitor_list.index', compact('accounts', 'currencies', 'date_from', 'date_to'));
    }

    public function indexcreditorList(Request $request)
    {
        $accounts = ChartOfAccounts::whereNotIn('level', [1, 2])->get();
        $date_from = Carbon::now()->month >= 7.
            ? Carbon::now()->year . '-07-01'
            : Carbon::now()->subYear()->year . '-07-01';
        $date_to = Carbon::now()->format('Y-m-d');
        $currencies = CurrencyMaster::where('status', 1)->get();

        return view('accounts_report.creditor_list.index', compact('accounts', 'currencies', 'date_from', 'date_to'));
    }
    public function indexpdcm(Request $request)
    {
        $banks = ChartOfAccounts::where('sub_category_id', 2)->get();

        $currencies = CurrencyMaster::where('status', 1)->get();
        $clients = Party::where('party_is', 'customer')->orWhere('party_is', 'customer_vendor')->get();

        return view('accounts_report.pdcm.index', compact('banks', 'currencies', 'clients'));
    }
    public function indexcashflow(Request $request)
    {
        $parent_accounts = ChartOfAccounts::whereIn('level', [3, 4])->get();
        $date_from = Carbon::now()->month >= 7.
            ? Carbon::now()->year . '-07-01'
            : Carbon::now()->subYear()->year . '-07-01';
        $date_to = Carbon::now()->format('Y-m-d');
        $currencies = CurrencyMaster::where('status', 1)->get();
        $clients = Party::where('party_is', 'customer')->orWhere('party_is', 'customer_vendor')->get();

        return view('accounts_report.cash_flow.index', compact('parent_accounts', 'currencies', 'clients', 'date_from', 'date_to'));
    }
    public function reportcashflow(Request $request)
    {
        $accountId = $request->input('account_id');
        $startDate = $request->input('date_from');
        $endDate = $request->input('date_to');

        // Calculate Opening Balance
        // $openingBalance = DB::table('transaction_detail')
        //     ->where('account_id', $accountId)
        //     ->where('created_at', '>', $startDate)
        //     ->whereNull('deleted_at')
        //     ->selectRaw('SUM(debit_lc) - SUM(credit_lc) as opening_balance')
        //     ->value('opening_balance');
        // $openingBalance = DB::table('chart_of_accounts_opening')
        //     ->where('chart_of_account_id', $accountId)
        //     ->selectRaw('SUM(debit_amount) - SUM(credit_amount) as opening_balance')
        //     ->value('opening_balance');
        $account = ChartOfAccounts::where('id', $accountId)->first();

        $openingBalance = $account->opening ? ($account->opening->debit_amount ? $account->opening->debit_amount : $account->opening->credit_amount) : 0;


        // Fetch Account In Transactions
        $accountIn = DB::table('transaction_detail as td_debit')
            ->join('transaction_master as tm', 'td_debit.transaction_master_id', '=', 'tm.id')
            ->join('transaction_detail as td_credit', function ($join) {
                $join->on('td_debit.transaction_master_id', '=', 'td_credit.transaction_master_id')
                    ->where('td_credit.credit_lc', '>', 0); // Match the credit row
            })
            ->join('chart_of_accounts as coa_debit', 'td_debit.account_id', '=', 'coa_debit.id') // Get details of the debit account
            ->join('chart_of_accounts as coa_credit', 'td_credit.account_id', '=', 'coa_credit.id') // Get details of the credit account
            ->leftJoin('chart_of_accounts as coa_credit_parent', 'coa_credit.parent_account_id', '=', 'coa_credit_parent.id') // Join to get credit parent name
            ->where('td_debit.account_id', $accountId) // Filter for the selected account
            ->where('td_credit.account_id', '!=', $accountId) // Ensure the debit row is for a different account
            ->whereBetween('tm.created_at', [$startDate, $endDate]) // Filter by date
            ->whereNull('td_debit.deleted_at')
            ->select(
                'coa_debit.name as AccountDebit', // The selected account name
                'coa_credit.name as AccountCredit', // The corresponding credit account name
                'coa_credit_parent.name as AccountCreditParent', // The corresponding credit account name
                'td_debit.debit_lc as AmountIn' // The debit amount
            )
            ->get();

        // Fetch Account Out Transactions
        $accountOut = DB::table('transaction_detail as td_credit')
            ->join('transaction_master as tm', 'td_credit.transaction_master_id', '=', 'tm.id')
            ->join('transaction_detail as td_debit', function ($join) {
                $join->on('td_credit.transaction_master_id', '=', 'td_debit.transaction_master_id')
                    ->where('td_debit.debit_lc', '>', 0); // Match the debit row
            })
            ->join('chart_of_accounts as coa_credit', 'td_credit.account_id', '=', 'coa_credit.id') // Get details of the credit account
            ->join('chart_of_accounts as coa_debit', 'td_debit.account_id', '=', 'coa_debit.id') // Get details of the debit account
            ->leftJoin('chart_of_accounts as coa_debit_parent', 'coa_debit.parent_account_id', '=', 'coa_debit_parent.id') // Join to get credit parent name
            ->where('td_credit.account_id', $accountId) // Filter for the selected account in the credit row
            ->where('td_debit.account_id', '!=', $accountId) // Ensure the debit row is for a different account
            ->whereBetween('tm.created_at', [$startDate, $endDate]) // Filter by date
            ->whereNull('td_credit.deleted_at')
            ->select(
                'coa_credit.name as AccountCredit', // The selected account name (credit)
                'coa_debit.name as AccountDebit', // The corresponding debit account name
                'coa_debit_parent.name as AccountDebitParent', // The corresponding credit account name
                'td_credit.credit_lc as AmountOut' // The credit amount
            )
            ->get();


        // dd($accountOut);

        return view('accounts_report.cash_flow.report', compact('openingBalance', 'accountIn', 'accountOut'));
    }

    public function indexchartofaccountlist(Request $request)
    {
        $parent_accounts = ChartOfAccounts::whereIn('level', [2, 3])->get();
        $category = AccountCategory::all();
        $sub_category = AccountSubCategory::all();
        $currencies = CurrencyMaster::where('status', 1)->get();
        $clients = Party::where('party_is', 'customer')->orWhere('party_is', 'customer_vendor')->get();

        return view('accounts_report.chart_of_account_list.index', compact('parent_accounts', 'currencies', 'clients', 'category', 'sub_category'));
    }
    public function indexcashbook(Request $request)
    {
        $cash_bank_accounts = ChartOfAccounts::where('parent_account_id', 29)->get();
        $date_from = Carbon::now()->month >= 7.
            ? Carbon::now()->year . '-07-01'
            : Carbon::now()->subYear()->year . '-07-01';
        $date_to = Carbon::now()->format('Y-m-d');

        return view('accounts_report.cash_book.index', compact('cash_bank_accounts', 'date_from', 'date_to'));
    }
    public function indexbugetvariance(Request $request)
    {
        $parent_accounts = ChartOfAccounts::whereIn('level', [2, 3, 4])->get();
        $date_from = Carbon::now()->month >= 7.
            ? Carbon::now()->year . '-07-01'
            : Carbon::now()->subYear()->year . '-07-01';
        $date_to = Carbon::now()->format('Y-m-d');
        $currencies = CurrencyMaster::where('status', 1)->get();

        return view('accounts_report.budget_variance_report.index', compact('parent_accounts', 'currencies', 'date_from', 'date_to'));
    }
    public function indexageingreport(Request $request)
    {
        // $parent_accounts = ChartOfAccounts::whereIn('level', [2, 3])->get();
        $parent_accounts = ChartOfAccounts::whereIn('level', [3, 4])->whereIn('sub_category_id', [3, 4])->get();
        $currencies = CurrencyMaster::where('status', 1)->get();
        $clients = Party::where('party_is', 'customer')->orWhere('party_is', 'customer_vendor')->get();
        $date_from = Carbon::now()->month >= 7.
            ? Carbon::now()->year . '-07-01'
            : Carbon::now()->subYear()->year . '-07-01';
        $date_to = Carbon::now()->format('Y-m-d');

        return view('accounts_report.ageing_report.index', compact('parent_accounts', 'currencies', 'clients', 'date_from', 'date_to'));
    }
    public function indexchequeprinting(Request $request)
    {
        $currencies = CurrencyMaster::where('status', 1)->get();
        $clients = Party::where('party_is', 'customer')->orWhere('party_is', 'customer_vendor')->get();
        $transaction_masters = TransactionMaster::whereNotNull('voucher_no')
            ->where('voucher_no', 'like', 'B%')
            ->get();
        $date_from = Carbon::now()->month >= 7.
            ? Carbon::now()->year . '-07-01'
            : Carbon::now()->subYear()->year . '-07-01';
        $date_to = Carbon::now()->format('Y-m-d');

        return view('accounts_report.cheque_printing.index', compact('currencies', 'clients', 'transaction_masters', 'date_from', 'date_to'));
    }

    public function indexinvoicesettelment(Request $request)
    {
        $parent_accounts = ChartOfAccounts::whereIn('level', [3, 4])->where('category_id', 5)->get();
        $currencies = CurrencyMaster::where('status', 1)->get();
        $date_from = Carbon::now()->month >= 7.
            ? Carbon::now()->year . '-07-01'
            : Carbon::now()->subYear()->year . '-07-01';
        $date_to = Carbon::now()->format('Y-m-d');

        return view('accounts_report.invoice_settelment.index', compact('parent_accounts', 'currencies', 'date_from', 'date_to'));
    }
    public function indexclearunclear(Request $request)
    {
        $parent_accounts = ChartOfAccounts::whereIn('level', [1, 2, 3])->get();
        $child_accounts = ChartOfAccounts::whereIn('level', [4])->get();
        $bank = ChartOfAccounts::where('sub_category_id', 2)->get();
        $date_from = Carbon::now()->month >= 7.
            ? Carbon::now()->year . '-07-01'
            : Carbon::now()->subYear()->year . '-07-01';
        $date_to = Carbon::now()->format('Y-m-d');

        return view('accounts_report.clear_unclear_report.index', compact('parent_accounts', 'date_from', 'date_to', 'bank', 'child_accounts'));
    }
    public function reportclearunclear(Request $request)
    {
        $dateFrom = $request->date_from;
        $dateTill = $request->date_to;
        $parent_account_id = $request->parent_account_id;
        $account_id = $request->child_account;
        $bank = $request->bank;
        $banks = ChartOfAccounts::where('sub_category_id', 2)->pluck('id');


        $invoices = TransactionMaster::with('detail', 'chart_of_account', 'chart_of_accounts', 'transaction', 'bill')->whereIn('account_id', $banks);

        if ($parent_account_id) {
            $invoices->where(function ($query) use ($parent_account_id) {
                $query->where('account_id', $parent_account_id)
                    ->orWhere('party_id', $parent_account_id);
            });
        }

        if ($account_id) {
            $invoices->where(function ($query) use ($account_id) {
                $query->where('account_id', $account_id)
                    ->orWhere('party_id', $account_id);
            });
        }

        if ($bank) {
            $invoices->where(function ($query) use ($bank) {
                $query->where('account_id', $bank)
                    ->orWhere('party_id', $bank);
            });
        }

        $invoices = $invoices->get();
        $export_type = $request->export_type; // Check if export is requested

        if ($export_type) {
            $fileName = 'clear_unclear_cheques_' . now()->format('Y-m-d') . '.' . ($export_type === 'csv' ? 'csv' : 'xlsx');
            return Excel::download(new ClearExport($invoices), $fileName);
        }
        // dd($invoices);
        return view('accounts_report.clear_unclear_report.report', compact('invoices'));
    }
}
