<?php

namespace App\Http\Controllers\Accounts;

use App\Http\Controllers\Controller;
use App\Models\AccountCategory;
use App\Models\AccountSubCategory;
use App\Models\ChartOfAccounts;
use App\Models\ChartOfAccountsOpening;
use App\Models\CurrencyMaster;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\Facades\DataTables;

class OpeningBalancesController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    // Render page (no heavy data here)
    public function index(Request $request)
    {
        // prebuild currency options once (no N+1 in the datatable rows)
        $currencies = CurrencyMaster::where('status', 1)->get(['id', 'name']);
        $currencyOptions = view('opening_balances._currency_options', compact('currencies'))->render();

        return view('opening_balances.index', compact('currencyOptions'));
    }

    // Server-side data for DataTables
    public function data(Request $request)
    {
        // Fetch currencies once
        $currencies = CurrencyMaster::where('status', 1)->get(['id', 'name']);

        $q = ChartOfAccounts::query()
            ->whereIn('chart_of_accounts.level', [3, 4])
            ->leftJoin('chart_of_accounts_opening as ob', 'ob.chart_of_account_id', '=', 'chart_of_accounts.id')
            ->select([
                'chart_of_accounts.id',
                'chart_of_accounts.account_code',
                'chart_of_accounts.name',
                'ob.debit_amount',
                'ob.credit_amount',
                'ob.currency_id',
                'ob.exchange_rate',
                'ob.debit_lc',
                'ob.credit_lc',
            ]);

        $num = function ($v, $dec = 2) {
            return is_null($v) ? '' : number_format((float) $v, $dec, '.', '');
        };

        return DataTables::of($q)
            ->addColumn('debit', function ($r) use ($num) {
                $val = $num($r->debit_amount, 2);
                return '<input type="number" step="0.01" value="' . $val . '" class="form-control debit" data-id="' . $r->id . '">';
            })
            ->addColumn('credit', function ($r) use ($num) {
                $val = $num($r->credit_amount, 2);
                return '<input type="number" step="0.01" value="' . $val . '" class="form-control credit" data-id="' . $r->id . '">';
            })
            ->addColumn('currency', function ($r) use ($currencies) {
                $html = '<select class="form-control currency" data-id="' . $r->id . '"><option value="">Please select</option>';
                foreach ($currencies as $c) {
                    $sel = ((int) $r->currency_id === (int) $c->id) ? ' selected' : '';
                    $html .= '<option value="' . $c->id . '"' . $sel . '>' . $c->name . '</option>';
                }
                $html .= '</select>';
                return $html;
            })
            ->addColumn('rate', function ($r) use ($num) {
                $val = $num($r->exchange_rate ?? 1, 4);
                return '<input type="number" step="0.0001" value="' . $val . '" class="form-control rate" data-id="' . $r->id . '">';
            })
            ->addColumn('debit_lc', function ($r) use ($num) {
                $val = $num($r->debit_lc, 2);
                return '<input type="number" step="0.01" value="' . $val . '" class="form-control debit_lc" data-id="' . $r->id . '">';
            })
            ->addColumn('credit_lc', function ($r) use ($num) {
                $val = $num($r->credit_lc, 2);
                return '<input type="number" step="0.01" value="' . $val . '" class="form-control credit_lc" data-id="' . $r->id . '">';
            })
            ->addColumn(
                'action',
                fn($r) =>
                '<button class="btn btn-sm btn-primary save-row" data-id="' . $r->id . '">Save</button>'
            )
            ->rawColumns(['debit', 'credit', 'currency', 'rate', 'debit_lc', 'credit_lc', 'action'])
            ->make(true);
    }
    public function totals()
    {
        // Calculate the totals for debit_amount, credit_amount, debit_lc, and credit_lc
        $totals = ChartOfAccounts::query()
            ->whereIn('chart_of_accounts.level', [3, 4])
            ->leftJoin('chart_of_accounts_opening as ob', 'ob.chart_of_account_id', '=', 'chart_of_accounts.id')
            ->selectRaw('
            SUM(ob.debit_amount) AS total_debit,
            SUM(ob.credit_amount) AS total_credit,
            SUM(ob.debit_lc) AS total_debit_lc,
            SUM(ob.credit_lc) AS total_credit_lc
        ')
            ->first(); // Get the first (and only) result with the summed totals

        // Format the totals as numbers with two decimal places (if not null)
        $num = function ($v, $dec = 2) {
            return is_null($v) ? 0 : number_format((float) $v, $dec, '.', '');
        };

        // Return the totals as a JSON response
        return response()->json([
            'total_debit' => $num($totals->total_debit),
            'total_credit' => $num($totals->total_credit),
            'total_debit_lc' => $num($totals->total_debit_lc),
            'total_credit_lc' => $num($totals->total_credit_lc)
        ]);
    }



    // Save a single row via AJAX
    public function saveRow(Request $request)
    {
        $data = $request->validate([
            'account_id' => 'required|integer|exists:chart_of_accounts,id',
            'debit_amount' => 'nullable|numeric',
            'credit_amount' => 'nullable|numeric',
            'currency_id' => 'nullable|integer|exists:currency_master,id',
            'exchange_rate' => 'nullable|numeric',
            'debit_lc' => 'nullable|numeric',
            'credit_lc' => 'nullable|numeric',
        ]);

        // Optional: enforce that only one of debit/credit is non-zero
        if (($data['debit_amount'] ?? 0) > 0 && ($data['credit_amount'] ?? 0) > 0) {
            return response()->json(['ok' => false, 'message' => 'Debit and Credit cannot both be non-zero.'], 422);
        }

        try {
            DB::transaction(function () use ($data) {
                ChartOfAccountsOpening::updateOrCreate(
                    ['chart_of_account_id' => $data['account_id']],
                    [
                        'debit_amount' => $data['debit_amount'] ?? 0,
                        'credit_amount' => $data['credit_amount'] ?? 0,
                        'currency_id' => $data['currency_id'] ?? null,
                        'exchange_rate' => $data['exchange_rate'] ?? 1,
                        'debit_lc' => $data['debit_lc'] ?? 0,
                        'credit_lc' => $data['credit_lc'] ?? 0,
                    ]
                );
            });

            return response()->json(['ok' => true]);
        } catch (Exception $e) {
            return response()->json(['ok' => false, 'message' => 'Save failed.'], 500);
        }
    }

    public function add()
    {
        $account_categories = AccountCategory::all();
        $account_sub_categories = AccountSubCategory::all();
        $chart_of_accounts = ChartOfAccounts::where('status', 1)->where('sub_category_id', 2)->get();

        return view('cheque_book_stock.add', compact('chart_of_accounts', 'account_categories', 'account_sub_categories'));
    }

    public function store(Request $request)
    {
        try {
            foreach ($request->account_id as $index => $account) {
                // if($index){
                // dd($index);
                // dd($request['debit_amount'][$index]);
                ChartOfAccountsOpening::updateOrCreate([
                    'chart_of_account_id' => $account
                ], [
                    'debit_amount' => $request['debit_amount'][$index],
                    'credit_amount' => $request['credit_amount'][$index],
                    'currency_id' => $request['currency_id'][$index],
                    'debit_lc' => $request['debit_lc'][$index],
                    'credit_lc' => $request['credit_lc'][$index],
                    'exchange_rate' => $request['exchange_rate'][$index]
                ]);
                // }

            }
        } catch (Exception $e) {
            dd($e);
        }
    }

    public function edit(ChartOfAccounts $account)
    {
        $account_categories = AccountCategory::all();
        $account_sub_categories = AccountSubCategory::all();
        $parent_accounts = ChartOfAccounts::whereNull('parent_account_id')->whereNot('id', $account->id)->get();

        return view('cheque_book_stock.edit', compact('account', 'account_categories', 'account_sub_categories', 'parent_accounts'));
    }

    public function update(ChartOfAccounts $account, Request $request)
    {
        try {

            ChartOfAccounts::where('id', $account->id)->update([
                'name' => $request->name,
                'account_code' => $request->account_code,
                'alias' => $request->alias,
                'max_child_account' => $request->max_child_account,
                'category_id' => $request->category_id,
                'sub_category_id' => $request->sub_category_id,
                // 'pl_category_id' => $request->name,
                'reference_number' => $request->reference_number,
                'parent_account_id' => $request->parent_account_id,
                'status' => $request->status
            ]);

            return redirect('accounts/cheque_book');
        } catch (Exception $e) {
            dd($e);
        }
    }

    public function delete(ChartOfAccounts $account)
    {
        ChartOfAccounts::where('id', $account->id)->update([
            'deleted_at' => now(),
            'deleted_by' => 1,
        ]);

        return redirect('accounts/cheque_book');
    }
}
