<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;

use App\Models\Airport;
use App\Http\Requests\AirportRequest;
use App\Models\BillCharges;
use App\Models\Bills;
use App\Models\BlContainerInfo;
use App\Models\BlMaster;
use App\Models\Charge;
use App\Models\ChartOfAccounts;
use App\Models\CurrencyMaster;
use App\Models\City;
use App\Models\Country;
use App\Models\JobChargesPayable;
use App\Models\JobMaster;
use App\Models\Operations;
use App\Models\Party;
use App\Models\TransactionDetail;
use App\Models\TransactionMaster;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Spatie\Browsershot\Browsershot;


class SEBillController extends Controller
{

    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(Request $request)
    {
        $perPage = $request->input('perPage', session('perPage', 100));
        session(['perPage' => $perPage]);
        // Get filter parameters from the request
        $filterBy = $request->filterBy;
        $filterValue = $request->filterValue;
        // dd($filterBy, $filterValue);
        $url = $request->path();
        if ($url == "se_bill") {
            $query = Bills::where('operation_id', 1);
        } elseif ($url == "si_bill") {
            $query = Bills::where('operation_id', 2);
        } elseif ($url == "ae_bill") {
            $query = Bills::where('operation_id', 3);
        } elseif ($url == "ai_bill") {
            $query = Bills::where('operation_id', 4);
        }

        if ($filterBy && $filterValue) {
            switch ($filterBy) {
                case 'id':
                    $bills = $query->where('id', $filterValue)->paginate($perPage);
                    break;
                case 'ref_bill_no':
                    $bills = $query->where('ref_bill_no', $filterValue)->paginate($perPage);
                    break;
                case 'vendor_id':
                    $bills = $query->whereHas('vendor', function ($q) use ($filterValue) {
                        $q->where('name', 'like', "%{$filterValue}%");
                    })->paginate($perPage);
                    break;
                case 'job_master_id':
                    $bills = $query->where('job_master_id', $filterValue)->paginate($perPage);
                    break;
                case 'invoice_type':
                    $bills = $query->where('invoice_type', $filterValue)->paginate($perPage);
                    break;
                case 'status':
                    $bills = $query->where('status', $filterValue)->paginate($perPage);
                    break;
                case 'vndr_tax_inv_no':
                    $bills = $query->where('vndr_tax_inv_no', $filterValue)->paginate($perPage);
                    break;
                case 'vndr_cmercial_inv_no':
                    $bills = $query->where('vndr_cmercial_inv_no', $filterValue)->paginate($perPage);
                    break;
                default:
                    return redirect()->back()->with('error', 'Invalid filter key: ' . $filterBy);
            }
        } else {
            $bills = $query->orderBy('id', 'Desc')->paginate($perPage);
        }

        return view('sea_export.se_bill.index', compact('bills', 'perPage'));
    }

    public function add(JobMaster $job_master)
    {
        $vendors = Party::where('party_is', 'vendor')->orWhere('party_is', 'customer_vendor')->get();
        $charges = Charge::all();
        $operations = Operations::all();
        $bill = Bills::where('job_master_id', $job_master->id)->first();

        if ($bill != null) {
            if ($job_master->operation_id == 1) {
                return redirect()->to('se_bill/edit/' . $job_master->id);
            } else if ($job_master->operation_id == 2) {
                return redirect()->to('si_bill/edit/' . $job_master->id);
            } else if ($job_master->operation_id == 3) {
                return redirect()->to('ae_bill/edit/' . $job_master->id);
            } else if ($job_master->operation_id == 4) {
                return redirect()->to('ai_bill/edit/' . $job_master->id);
            }
        } else {
            return view('sea_export.se_bill.add', compact('operations', 'vendors', 'charges', 'job_master'));
        }
    }

    public function autoBill(JobMaster $job_master, Request $request, LogsController $logController)
    {
        try {
            $job_charges_payable = JobChargesPayable::whereIn('id', $request->job_payable_charges_ids)->whereNull('bill_invoice')->get();
            if ($job_charges_payable->isEmpty()) {
                return response()->json(['error' => 'There are no Payable exist', 'status' => 500], 500);
            } else {
                DB::beginTransaction();

                $chargesGroupedByVendor = $job_charges_payable->groupBy('vendor_id');
                foreach ($chargesGroupedByVendor as $vendor_id => $charges) {
                    if ($vendor_id) {
                        $party_id = ChartOfAccounts::where('party_id', $vendor_id)->first();

                        // dd($latest_bill);
                        if ($party_id) {
                            // $chargesGroupedByCurrency = $charges->groupBy('currency');
                            // foreach ($chargesGroupedByCurrency as $currency => $currencyCharges) {
                            // $latest_bill = Bills::latest()->first();
                            $lastId = Bills::orderByDesc('id')->lockForUpdate()->value('id') ?? 0;
                            $nextSeq = $lastId + 1;
                            $sum_amount = $charges->sum('amount');
                            $sum_discount = $charges->sum('discount');
                            $sum_tax = $charges->sum('tax_amount_lc');
                            $sum_net = $charges->sum('net_amount');
                            $sum_local = $charges->sum('local_amount');
                            $bill = Bills::create([
                                'job_master_id' => $job_master->id,
                                'tran_no' => 'JB-' . $nextSeq . '/' . date('y'),
                                'bill_date' => $job_master->job_date ?? date('Y-m-d'),
                                'reference' => $job_master->file_no,
                                'settlement' => "un_settled",
                                'vendor_id' => $vendor_id,
                                'sequence_no' => $request->sequence_no,
                                'invoice_type' => "PI",
                                'ref_bill_no' => $request->ref_bill_no,
                                'operation_id' => $job_master->operation_id,
                                'aut_man' => $request->aut_man,
                                'cost_center' => "Head Office",
                                'single_multiple' => $request->single_multiple,
                                'auto_round_off' => $request->auto_round_off,
                                'rebate' => $request->rebate,
                                'manual' => $request->manual,
                                'continue_mode' => $request->continue_mode,
                                'show_terminal' => $request->show_terminal,
                                'due_days' => $request->due_days,
                                'vndr_tax_inv_no' => $request->vndr_tax_inv_no,
                                'vndr_cmercial_inv_no' => $request->vndr_cmercial_inv_no,
                                'vndr_inv_date' => $request->vndr_inv_date,
                                'remarks' => $request->remarks,
                                'voucher_no' => $request->voucher_no,
                                'bank_details' => $request->bank_details,
                                'manual_remarks' => $request->manual_remarks,
                                // 'grand_total_amount' => $charges->sum('amount'),
                                // 'grand_net_amount' => $charges->sum('amount'),
                                // 'grand_discount' => $charges->sum('discount'),
                                // 'grand_tax_amount' => $charges->sum('tax_amount_lc'),
                                // 'grand_net_amount_inc_tax' => $charges->sum('net_amount'),
                                // 'grand_local_amount' => $charges->sum('local_amount'),
                                'grand_total_amount' => $sum_amount,
                                'grand_net_amount' => $sum_amount,
                                'grand_discount' => $sum_discount,
                                'grand_tax_amount' => $sum_tax,
                                'grand_net_amount_inc_tax' => $sum_net,
                                'grand_local_amount' => $sum_local,
                                'total_deduction' => $request->total_deduction,
                                'status' => 1,
                            ]);

                            $last_voucher_number = TransactionMaster::where('voucher_type_id', 9)
                                ->max('voucher_no');


                            if (preg_match('/-(\d{5})\//', $last_voucher_number, $matches)) {
                                $sequence_number = intval($matches[1]) + 1;
                            } else {
                                $sequence_number = 1;
                            }
                            $sequence_number_padded = str_pad($sequence_number, 5, '0', STR_PAD_LEFT);
                            $year = date('y', strtotime($bill->bill_date ?? date('Y-m-d')));
                            $new_voucher_number = 'PI-' . $sequence_number_padded . '/' . $year;

                            $bill->update(['voucher_no' => $new_voucher_number]);

                            $transaction_master = TransactionMaster::create([
                                'voucher_no' => $new_voucher_number,
                                'bill_id' => $bill->id,
                                'job_master_id' => $bill->job_master_id,
                                'date' => date('Y-m-d'),
                                'transaction_type_master_id' => 4,
                                'account_id' => 153, // Credit
                                'cost_center_id' => 1,
                                'party_id' => $party_id->id,
                                // 'currency_id' => 1,
                                'exchange_rate' => $job_master->selling,
                                // 'cheque_date' => $request->cheque_date,
                                // 'total_debit' => $charges->sum('local_amount'),
                                // 'total_credit' => $charges->sum('local_amount'),
                                'total_debit_converted' => $sum_local,
                                'total_credit_converted' => $sum_local,
                                'total_debit' => $sum_net,
                                'total_credit' => $sum_net,
                                'currency_id' => 1,
                                // 'bank_sub_type' => $request->bank_sub_type,
                                'company_id' => 1,
                                'status' => 4,
                                // 'narration' => 'Payable Against Job # ' . 'PI-' . ((int) ($latest_bill->id ?? 0) + 1) . '/' . date('y') . ' From ' . $party_id->name . '',
                                'narration' => 'Payable Against Job # ' . 'PI-' . $nextSeq . '/' . date('y') . ', HBL Number: ' . $job_master->hbl_number . ', Reference Number: ' . $job_master->file_no . ', Bill #: ' . $bill->tran_no . ' To ' . $party_id->name . '',
                                'created_by' => auth()->user()->id,
                                'voucher_type_id' => 9
                            ]);

                            // Debit
                            TransactionDetail::create([
                                'transaction_master_id' => $transaction_master->id,
                                'account_id' => $party_id->id,
                                'job_master_id' => $bill->job_master_id,
                                // 'cost_center_id' => $request->cost_center_detail[$key],
                                'credit_vc' => $sum_net,
                                'debit_vc' => 0,
                                'credit_lc' => $sum_local,
                                'debit_lc' => 0,
                                // 'narration' => 'Payable Against Job # ' . 'PI-' . ((int) ($latest_bill->id ?? 0) + 1) . '/' . date('y') . ' From ' . $party_id->name . '',
                                'narration' => 'Payable Against Job # ' . 'PI-' . $nextSeq . '/' . date('y') . ', HBL Number: ' . $job_master->hbl_number . ', Reference Number: ' . $job_master->file_no . ', Bill #: ' . $bill->tran_no . ' To ' . $party_id->name . '',
                                'status' => 1
                            ]);

                            // Credit
                            TransactionDetail::create([
                                'transaction_master_id' => $transaction_master->id,
                                'account_id' => 153,
                                'job_master_id' => $bill->job_master_id,
                                // 'cost_center_id' => $request->cost_center_detail[$key],
                                'credit_vc' => 0,
                                'debit_vc' => $sum_net,
                                'credit_lc' => 0,
                                'debit_lc' => $sum_local,
                                // 'narration' => 'Payable Against Job # ' . 'PI-' . ((int) ($latest_bill->id ?? 0) + 1) . '/' . date('y') . ' From ' . $party_id->name . '',
                                'narration' => 'Payable Against Job # ' . 'PI-' . $nextSeq . '/' . date('y') . ', HBL Number: ' . $job_master->hbl_number . ', Reference Number: ' . $job_master->file_no . ', Bill #: ' . $bill->tran_no . ' To ' . $party_id->name . '',
                                'status' => 1
                            ]);

                            foreach ($charges as $detail) {
                                BillCharges::create([
                                    'bill_id' => $bill->id,
                                    'transaction_master_id' => $transaction_master->id,
                                    'job_charge_payable_id' => $detail->id,
                                    'job_master_id' => $job_master->id,
                                    'charge_id' => $detail->charge_id,
                                    'description' => $detail->description,
                                    'size_type' => $detail->size_type,
                                    'rate_group' => $detail->rate_group,
                                    'dg_non_dg' => $detail->dg_non_dg,
                                    'container' => $detail->container,
                                    'qty' => $detail->qty,
                                    'rate' => $detail->rate,
                                    'currency' => $detail->currency,
                                    'amount' => $detail->amount,
                                    'discount' => $detail->discount,
                                    'net_amount' => $detail->net_amount,
                                    'tax' => $detail->tax,
                                    'tax_amount' => $detail->tax_amount,
                                    'net_amount_inc_tax' => $detail->net_amount_inc_tax,
                                    'ex_rate' => $detail->ex_rate,
                                    'local_amount' => $detail->local_amount,
                                    'status' => 1,
                                ]);

                                JobChargesPayable::where('id', $detail->id)->update([
                                    'bill_invoice' => $bill->tran_no
                                ]);
                            }
                            // }
                        } else {
                            // dd($vendor_id);
                            DB::rollBack();

                            $logController->createLog(__METHOD__, 'error', 'Chart Of Account Not Found', auth()->user(), '');

                            return response()->json(['error' => 'Chart Of Account Not Found', 'status' => 500], 500);
                        }
                    } else {
                        DB::rollBack();

                        $logController->createLog(__METHOD__, 'error', 'Vendor Name not selected', auth()->user(), '');

                        return response()->json(['error' => 'Vendor Name not selected', 'status' => 500], 500);
                    }
                }
                DB::commit();
                $logController->createLog(__METHOD__, 'success', 'SE Bill Created.', auth()->user(), '');
                session()->flash('success', 'Bill Created Successfully');
                return response()->json(['data' => 'Data Saved!', 'status' => 200], 200);
            }
        } catch (Exception $e) {
            // dd($e);
            DB::rollBack();

            $logController->createLog(__METHOD__, 'error', $e, auth()->user(), '');

            return response()->json(['data' => $e, 'status' => 500], 500);
        }
    }
    public function manualBill(JobMaster $job_master, Request $request, LogsController $logController)
    {
        try {
            // $job_charges_payable = JobChargesPayable::whereIn('id', $request->job_payable_charges_ids)->get();
            $latest_bill = Bills::latest()->first();
            $voucher_no = 'PI-' . ((int) ($latest_bill->id ?? 0) + 1) . '/' . date('y');

            $job_charges_payable = JobChargesPayable::whereNull('bill_invoice')
                ->whereIn('id', explode(',', $request->job_payable_charges_ids))
                ->get();
            $charges_ids = $request->job_payable_charges_ids;
            $charges = Charge::all();
            $vendors = Party::where('party_is', 'vendor')->orWhere('party_is', 'customer_vendor')->get();
            $operations = Operations::all();
            $currency = CurrencyMaster::all();
            $grand_total_amount = $job_charges_payable->sum('amount');
            $grand_net_amount = $job_charges_payable->sum('amount');
            $grand_discount = $job_charges_payable->sum('discount');
            $grand_tax_amount = $job_charges_payable->sum('tax_amount_lc');
            $grand_net_amount_inc_tax = $job_charges_payable->sum('net_amount');
            $grand_local_amount = $job_charges_payable->sum('local_amount');
            // dd($job_charges_payable);
            return view('sea_export.se_bill.addFromJobManual', compact('voucher_no', 'charges_ids', 'job_master', 'job_charges_payable', 'latest_bill', 'vendors', 'operations', 'charges', 'currency', 'grand_total_amount', 'grand_net_amount', 'grand_discount', 'grand_tax_amount', 'grand_net_amount_inc_tax', 'grand_local_amount'));
        } catch (Exception $e) {

            $logController->createLog(__METHOD__, 'error', $e, auth()->user(), '');

            return redirect()->back()->with('error', 'Something Went Wrong! ' . $e->getMessage());
        }
    }

    public function store(JobMaster $job_master, Request $request, LogsController $logController)
    {
        try {

            $party_id = ChartOfAccounts::where('party_id', $request->vendor_id)->first();
            if ($party_id) {

                $latest_bill = Bills::latest()->first();
                $bill = Bills::create([
                    'job_master_id' => $job_master->id,
                    'tran_no' => $request->tran_no,
                    'bill_date' => $request->bill_date,
                    'reference' => $request->reference,
                    'settlement' => $request->settlement,
                    'vendor_id' => $request->vendor_id,
                    'sequence_no' => $request->sequence_no,
                    'invoice_type' => $request->invoice_type,
                    'ref_bill_no' => $request->ref_bill_no,
                    'operation_id' => $request->operation_id,
                    'aut_man' => $request->aut_man,
                    'cost_center' => $request->cost_center,
                    'single_multiple' => $request->single_multiple,
                    'auto_round_off' => $request->auto_round_off,
                    'rebate' => $request->rebate,
                    'manual' => $request->manual,
                    'continue_mode' => $request->continue_mode,
                    'show_terminal' => $request->show_terminal,
                    'due_days' => $request->due_days,
                    'vndr_tax_inv_no' => $request->vndr_tax_inv_no,
                    'vndr_cmercial_inv_no' => $request->vndr_cmercial_inv_no,
                    'vndr_inv_date' => $request->vndr_inv_date,
                    'remarks' => $request->remarks,
                    'voucher_no' => $request->voucher_no,
                    'bank_details' => $request->bank_details,
                    'manual_remarks' => $request->manual_remarks,
                    'grand_total_amount' => $request->grand_total_amount,
                    'grand_net_amount' => $request->grand_net_amount,
                    'grand_discount' => $request->grand_discount,
                    'grand_tax_amount' => $request->grand_tax_amount,
                    'grand_net_amount_inc_tax' => $request->grand_net_amount_inc_tax,
                    'grand_local_amount' => $request->grand_local_amount,
                    'total_deduction' => $request->total_deduction,
                    'status' => 1,
                ]);

                $last_voucher_number = TransactionMaster::where('voucher_type_id', 9)
                    ->max('voucher_no');


                if (preg_match('/-(\d{5})\//', $last_voucher_number, $matches)) {
                    $sequence_number = intval($matches[1]) + 1;
                } else {
                    $sequence_number = 1;
                }
                $sequence_number_padded = str_pad($sequence_number, 5, '0', STR_PAD_LEFT);
                $year = date('y', strtotime($bill->bill_date ?? date('Y-m-d')));
                $new_voucher_number = 'PI-' . $sequence_number_padded . '/' . $year;

                $bill->update(['voucher_no' => $new_voucher_number]);

                $transaction_master = TransactionMaster::create([
                    'job_master_id' => $bill->job_master_id,
                    'voucher_no' => $new_voucher_number,
                    'bill_id' => $bill->id,
                    'date' => date('Y-m-d'),
                    'transaction_type_master_id' => 4,
                    'account_id' => 153, // Credit
                    'cost_center_id' => 1,
                    'party_id' => $party_id->id,
                    'currency_id' => 1,
                    'exchange_rate' => $job_master->selling,
                    // 'cheque_date' => $request->cheque_date,
                    'total_debit_converted' => $request->grand_local_amount,
                    'total_credit_converted' => $request->grand_local_amount,
                    'total_debit' => $request->grand_net_amount_inc_tax,
                    'total_credit' => $request->grand_net_amount_inc_tax,
                    // 'bank_sub_type' => $request->bank_sub_type,
                    'company_id' => 1,
                    'status' => 4,
                    'narration' => 'Payable Against Job # ' . 'PI-' . ((int) ($latest_bill->id ?? 0) + 1) . '/' . date('y') . ' To ' . $party_id->name . '',
                    'created_by' => auth()->user()->id,
                    'voucher_type_id' => 9
                ]);

                // Debit
                TransactionDetail::create([
                    'transaction_master_id' => $transaction_master->id,
                    'account_id' => $party_id->id,
                    // 'cost_center_id' => $request->cost_center_detail[$key],
                    'job_master_id' => $bill->job_master_id,
                    'credit_vc' => $request->grand_net_amount_inc_tax,
                    'debit_vc' => 0,
                    'credit_lc' => $request->grand_local_amount,
                    'debit_lc' => 0,
                    'narration' => 'Payable Against Job # ' . 'PI-' . ((int) ($latest_bill->id ?? 0) + 1) . '/' . date('y') . ' To ' . $party_id->name . '',
                    'status' => 1
                ]);

                // Credit
                TransactionDetail::create([
                    'transaction_master_id' => $transaction_master->id,
                    'account_id' => 153,
                    // 'cost_center_id' => $request->cost_center_detail[$key],
                    'job_master_id' => $bill->job_master_id,
                    'credit_vc' => 0,
                    'debit_vc' => $request->grand_net_amount_inc_tax,
                    'credit_lc' => 0,
                    'debit_lc' => $request->grand_local_amount,
                    'narration' => 'Payable Against Job # ' . 'PI-' . ((int) ($latest_bill->id ?? 0) + 1) . '/' . date('y') . ' To ' . $party_id->name . '',
                    'status' => 1
                ]);

                foreach ($request->charge_id as $index => $detail) {
                    $bill_charges = BillCharges::create([
                        'bill_id' => $bill->id,
                        'transaction_master_id' => $transaction_master->id,
                        'charge_id' => $detail,
                        'job_master_id' => $bill->job_master_id,
                        'job_charge_payable_id' => @$request->job_charge_payable_id[$index],
                        'description' => $request->description[$index],
                        'size_type' => $request->size_type[$index],
                        'rate_group' => $request->rate_group[$index],
                        'dg_non_dg' => $request->dg_non_dg[$index],
                        'container' => $request->container[$index],
                        'qty' => $request->qty[$index],
                        'rate' => $request->rate[$index],
                        'currency' => $request->currency[$index],
                        'amount' => $request->amount[$index],
                        'discount' => $request->discount[$index],
                        'net_amount' => $request->net_amount[$index],
                        'tax' => $request->tax[$index],
                        'tax_amount' => $request->tax_amount[$index],
                        'net_amount_inc_tax' => $request->net_amount_inc_tax[$index],
                        'ex_rate' => $request->ex_rate[$index],
                        'local_amount' => $request->local_amount[$index],
                        'status' => 1,
                    ]);
                    if (isset($request->job_charge_payable_id[$index])) {
                        JobChargesPayable::where('id', $request->job_charge_payable_id[$index])->update([
                            'bill_invoice' => $bill->tran_no
                        ]);
                    }
                }
                $logController->createLog(__METHOD__, 'success', 'SE Bill Created.', auth()->user(), '');

                // return redirect()->to('/se_bill')->with('success', 'New Record Created SuccessFully!');
                if ($job_master->operation_id == 1) {
                    return redirect()->to('/se_bill/edit/' . $job_master->id)->with('success', 'Record Created SuccessFully!');
                } else if ($job_master->operation_id == 2) {
                    return redirect()->to('/si_bill/edit/' . $job_master->id)->with('success', 'Record Created SuccessFully!');
                } else if ($job_master->operation_id == 3) {
                    return redirect()->to('/ae_bill/edit/' . $job_master->id)->with('success', 'Record Created SuccessFully!');
                } else if ($job_master->operation_id == 4) {
                    return redirect()->to('/ai_bill/edit/' . $job_master->id)->with('success', 'Record Created SuccessFully!');
                }
            } else {
                $logController->createLog(__METHOD__, 'error', 'Chart Of Account Not Found', auth()->user(), '');

                return redirect()->back()->with('error', 'Chart Of Account Not Found');
            }
        } catch (Exception $e) {

            $logController->createLog(__METHOD__, 'error', $e, auth()->user(), '');

            return redirect()->back()->with('error', 'Something Went Wrong! ' . $e->getMessage());
        }
    }

    public function edit(JobMaster $job_master, Request $request)
    {
        if ($request->bill_invoice) {
            $bill = Bills::where('job_master_id', $job_master->id)
                ->where('tran_no', $request->bill_invoice)
                ->first();
            // dd($bill);
            if (!$bill) {
                $bill = Bills::where('previous_job_master_id', $job_master->id)
                    ->where('tran_no', $request->bill_invoice)
                    ->first();
            }
        } else {
            $bill = Bills::where('job_master_id', $job_master->id)->first();
        }
        // dd($bill);
        $transaction_master = TransactionMaster::where('bill_id', $bill->id)->first();
        $vendors = Party::where('party_is', 'vendor')->orWhere('party_is', 'customer_vendor')->get();
        $charges = Charge::all();
        $operations = Operations::all();

        return view('sea_export.se_bill.edit', compact('bill', 'operations', 'vendors', 'charges', 'job_master', 'transaction_master'));
    }

    public function update(JobMaster $job_master, Request $request, LogsController $logController)
    {
        try {
            $bill = Bills::where('job_master_id', $job_master->id)->where('tran_no', $request->tran_no)->first();

            $status = 1;
            if ($request->status == 2) {
                $status = 0;
                // Store the current job_master_id as the previous job_master_id before setting it to null
                $previousJob_master_id = $job_master->id;
            } else {
                $status = $bill->status != 0 ? $bill->status : 1;
                // Use the previous_job_master_id if status is re-activated
                $previousJob_master_id = $bill->previous_job_master_id ?? $job_master->id;
            }
            Bills::where('job_master_id', $job_master->id)
                ->where('tran_no', $request->tran_no)
                ->update([
                    // 'job_master_id' => $request->status == 2 ? null : $previousJob_master_id,
                    'job_master_id' => $request->status == 2 ? null : $previousJob_master_id,
                    'previous_job_master_id' => $request->status == 2 ? $previousJob_master_id : null,
                    // 'tran_no' => $request->tran_no,
                    // 'bill_date' => $request->bill_date,
                    'reference' => $request->reference,
                    'status' => $status,
                    'settlement' => $request->settlement,
                    // 'vendor_id' => $request->vendor_id,
                    'sequence_no' => $request->sequence_no,
                    'invoice_type' => $request->invoice_type,
                    'ref_bill_no' => $request->ref_bill_no,
                    // 'operation_id' => $request->operation_id,
                    'aut_man' => $request->aut_man,
                    'cost_center' => $request->cost_center,
                    'single_multiple' => $request->single_multiple,
                    'auto_round_off' => $request->auto_round_off,
                    'rebate' => $request->rebate,
                    'manual' => $request->manual,
                    'continue_mode' => $request->continue_mode,
                    'show_terminal' => $request->show_terminal,
                    'due_days' => $request->due_days,
                    'vndr_tax_inv_no' => $request->vndr_tax_inv_no,
                    'vndr_cmercial_inv_no' => $request->vndr_cmercial_inv_no,
                    'vndr_inv_date' => $request->vndr_inv_date,
                    // 'remarks' => $request->remarks,
                    // 'voucher_no' => $request->voucher_no,
                    // 'bank_details' => $request->bank_details,
                    // 'manual_remarks' => $request->manual_remarks,
                    // 'grand_total_amount' => $request->grand_total_amount,
                    // 'grand_net_amount' => $request->grand_net_amount,
                    // 'grand_discount' => $request->grand_discount,
                    // 'grand_tax_amount' => $request->grand_tax_amount,
                    // 'grand_net_amount_inc_tax' => $request->grand_net_amount_inc_tax,
                    // 'grand_local_amount' => $request->grand_local_amount,
                    // 'total_deduction' => $request->total_deduction,
                ]);

            if ($request->status == 2) {
                JobChargesPayable::where('bill_invoice', $request->tran_no)->update([
                    'bill_invoice' => null,
                    'previous_bill_invoice' => $request->tran_no
                ]);
            } else if ($bill->previous_job_master_id != null) {
                JobChargesPayable::where('previous_bill_invoice', $request->tran_no)->update([
                    'bill_invoice' => $request->tran_no,
                    'previous_bill_invoice' => null,
                ]);
            }

            if ($request->status == 2) {
                $transaction_master = TransactionMaster::where('bill_id', $bill->id)->first();

                if ($transaction_master) {
                    TransactionDetail::where('transaction_master_id', $transaction_master->id)->delete();
                    $transaction_master->delete();
                }
                BillCharges::where('bill_id', $bill->id)->delete();
            } else if ($bill->previous_job_master_id != null) {
                $transaction_master = TransactionMaster::withTrashed()->where('bill_id', $bill->id)->first();
                BillCharges::withTrashed()->where('bill_id', $bill->id)->restore();

                if ($transaction_master) {
                    // Restore the TransactionMaster record
                    $transaction_master->restore();

                    // Retrieve and restore all associated soft-deleted TransactionDetail records
                    TransactionDetail::withTrashed()
                        ->where('transaction_master_id', $transaction_master->id)
                        ->restore();
                }
            }
            // if ($request->status == 2) {
            //     JobChargesPayable::where('bill_invoice', $request->tran_no)->update([
            //         'bill_invoice' => null
            //     ]);
            // }

            // if ($request->status == 2) {
            //     $transaction_master = TransactionMaster::where('bill_id', $bill->id)->first();

            //     if ($transaction_master) {
            //         TransactionDetail::where('transaction_master_id', $transaction_master->id)->delete();
            //         $transaction_master->delete();
            //     }
            // }

            // BillCharges::where('bill_id', $bill->id)->delete();

            // foreach ($request->charge_id as $index => $detail) {
            //     $bill_charges = BillCharges::create([
            //         'bill_id' => $bill->id,
            //         'charge_id' => $detail,
            //         'job_charge_payable_id' => $request->job_charge_payable_id[$index],
            //         'description' => $request->description[$index],
            //         'size_type' => $request->size_type[$index],
            //         'rate_group' => $request->rate_group[$index],
            //         'dg_non_dg' => $request->dg_non_dg[$index],
            //         'container' => $request->container[$index],
            //         'qty' => $request->qty[$index],
            //         'rate' => $request->rate[$index],
            //         'currency' => $request->currency[$index],
            //         'amount' => $request->amount[$index],
            //         'discount' => $request->discount[$index],
            //         'net_amount' => $request->net_amount[$index],
            //         'tax' => $request->tax[$index],
            //         'tax_amount' => $request->tax_amount[$index],
            //         'net_amount_inc_tax' => $request->net_amount_inc_tax[$index],
            //         'ex_rate' => $request->ex_rate[$index],
            //         'local_amount' => $request->local_amount[$index],
            //         'status' => 1,
            //     ]);
            // }
            $logController->createLog(__METHOD__, 'success', 'SE Bill Updated.', auth()->user(), '');

            // return redirect()->to('/se_bill')->with('success', ' Record Updated SuccessFully!');
            if ($job_master->operation_id == 1) {
                return redirect()->to('/se_bill/edit/' . $job_master->id . '?bill_invoice=' . $request->tran_no)->with('success', 'Record Updated SuccessFully!');
            } else if ($job_master->operation_id == 2) {
                return redirect()->to('/si_bill/edit/' . $job_master->id . '?bill_invoice=' . $request->tran_no)->with('success', 'Record Updated SuccessFully!');
            } else if ($job_master->operation_id == 3) {
                return redirect()->to('/ae_bill/edit/' . $job_master->id . '?bill_invoice=' . $request->tran_no)->with('success', 'Record Updated SuccessFully!');
            } else if ($job_master->operation_id == 4) {
                return redirect()->to('/ai_bill/edit/' . $job_master->id . '?bill_invoice=' . $request->tran_no)->with('success', 'Record Updated SuccessFully!');
            }
        } catch (Exception $e) {
            dd($e);

            $logController->createLog(__METHOD__, 'error', $e, auth()->user(), '');

            return redirect()->back()->with('error', 'Something Went Wrong!');
        }
    }

    public function print(Bills $bill, LogsController $logController, Request $request)
    {
        try {
            $pdf = $request->pdf;
            $print_logo = $request->print_logo;
            $print_on_letterhead = $request->print_on_letterhead;
            $print_container_sizetype = $request->print_container_sizetype;
            $print_container_no = $request->print_container_no;
            $add_description_with_charges = $request->add_description_with_charges;
            $print_remarks = $request->print_remarks;
            $print_sales_rep = $request->print_sales_rep;
            $add_description_only = $request->add_description_only;
            $hide_shipping_line = $request->hide_shipping_line;

            $bl_master = BlMaster::where('job_master_id', $bill->job_master_id)->first();
            $bl_container_info = $bl_master ? BLContainerInfo::where('bl_master_id', $bl_master->id)->first() : null;
            $containerNo = '';

            if ($bl_master && $bl_master->bl_container_info) {
                foreach ($bl_master->bl_container_info as $container) {
                    $containerNo = $bl_master->bl_container_info->pluck('container_no')->implode(', ');
                }
            }
            // if ($request->print_option === 'download_pdf') {
            //     // dd($bill);
            //     $html = view('sea_export.se_bill.pdf', compact(

            //         'containerNo',
            //         'print_logo',
            //         'print_on_letterhead',
            //         'print_container_sizetype',
            //         'print_container_no',
            //         'add_description_with_charges',
            //         'print_remarks',
            //         'print_sales_rep',
            //         'add_description_only',
            //         'hide_shipping_line',
            //         'bill'
            //     ))->render();
            //     // dd($html);
            //     try {
            //         return response()->streamDownload(function () use ($html) {
            //             echo Browsershot::html($html)
            //                 ->setOption('args', ['--no-sandbox'])
            //                 ->waitUntilNetworkIdle()
            //                 ->timeout(60) // Add timeout in seconds
            //                 ->format('A4')
            //                 ->margins(10, 10, 10, 10)
            //                 ->setOption('printBackground', true)
            //                 ->pdf();
            //         }, 'invoice.pdf');
            //         $logController->createLog(__METHOD__, 'success', 'PDF generated', auth()->user(), '');
            //     } catch (\Throwable $e) {

            //         $logController->createLog(__METHOD__, 'error', 'PDF generation failed', auth()->user(), $e->getMessage());

            //         return response()->json(['error' => 'PDF generation failed. Check logs.'], 500);
            //     }
            // }
            // dd($print_logo);
            if ($request->print_option === 'download_word') {
                $fileName = 'YARAN - JOB BILL - PRINT' . '.doc';
                $filePath = storage_path("app/public/{$fileName}");
                $html = view('sea_export.se_bill.word', compact('bill', 'print_logo', 'print_on_letterhead', 'print_container_sizetype', 'print_container_no', 'add_description_with_charges', 'print_remarks', 'print_sales_rep', 'add_description_only', 'hide_shipping_line', 'containerNo'))->render();
                // Add HTML wrapper and use inline CSS-friendly layout
                $wrappedHtml = '<html>
            <head>
                <meta charset="utf-8" />
                <meta name="viewport" content="width=device-width, initial-scale=1.0" />
                <meta name="ProgId" content="Word.Document" />
                <style>
                    body { font-family: Arial, sans-serif; font-size: 12px; }
                    table, th, td { border: 1px solid black; border-collapse: collapse; }
                    th, td { padding: 4px; }
                    img { max-width: 100%; }
                </style>
            </head>
            <body>' . $html . '</body>
        </html>';

                file_put_contents($filePath, $wrappedHtml);

                return response()->download($filePath)->deleteFileAfterSend(true);
            }
            $logController->createLog(__METHOD__, 'success', 'Invoice Print.', auth()->user(), '');

            return view('sea_export.se_bill.pdf', compact('bill', 'print_logo', 'print_on_letterhead', 'print_container_sizetype', 'print_container_no', 'add_description_with_charges', 'print_remarks', 'print_sales_rep', 'add_description_only', 'hide_shipping_line', 'containerNo'));
        } catch (Exception $e) {
            // dd($e);
            $logController->createLog(__METHOD__, 'error', $e, auth()->user(), $e->getMessage());

            return redirect()->back()->with('error', 'Something Went Wrong!');
        }
    }
    public function export($id, Request $request)
    {
        $invoice = Bills::findOrFail($id);
        $type = $request->type;
        $logoPath = public_path('logo2.png');
        $bl_master = BlMaster::where('job_master_id', $invoice->job_master->id)->first();
        $bl_container_info = $bl_master ? BlContainerInfo::where('bl_master_id', $bl_master->id)->first() : null;
        $containerNo = '';


        if ($bl_master && $bl_master->bl_container_info) {
            foreach ($bl_master->bl_container_info as $container) {
                $containerNo = $bl_master->bl_container_info->pluck('container_no')->implode(', ');
            }
        }
        // dd($containerNo);
        $data = [
            'invoice' => $invoice,
            'logoBase64' => 'data:image/png;base64,' . base64_encode(file_get_contents($logoPath)),
        ];

        // Render the Blade view into HTML
        $html = view('sea_export.se_invoice.pdf', $data)->render();



        // 📝 WORD EXPORT (.doc with full HTML + inline logo)
        if ($type === 'word') {
            $fileName = 'invoice_' . time() . '.doc';
            $filePath = storage_path("app/public/{$fileName}");

            // Add HTML wrapper and use inline CSS-friendly layout
            $wrappedHtml = '<html>
            <head>
                <meta charset="utf-8" />
                <meta name="viewport" content="width=device-width, initial-scale=1.0" />
                <meta name="ProgId" content="Word.Document" />
                <style>
                    body { font-family: Arial, sans-serif; font-size: 12px; }
                    table, th, td { border: 1px solid black; border-collapse: collapse; }
                    th, td { padding: 4px; }
                    img { max-width: 100%; }
                </style>
            </head>
            <body>' . $html . '</body>
        </html>';

            file_put_contents($filePath, $wrappedHtml);

            return response()->download($filePath)->deleteFileAfterSend(true);
        }

        // // 📊 EXCEL EXPORT (.xls with same layout)
        // if ($type === 'excel') {
        //     return Excel::download(new InvoiceExport($data), 'invoice.xlsx');
        // }

        return redirect()->back()->with('error', 'Invalid export type.');
    }
    public function getBillsByVendor($vendor_id)
    {
        $bills = Bills::with(['vendor', 'receipt_payment_details'])
            ->where('vendor_id', $vendor_id)
            ->get()
            ->map(function ($bill) {
                // Safely cast grand_local_amount to float
                $grandAmount = (float) str_replace(',', '', $bill->grand_local_amount);

                // Sum only numeric rcvd_amount values
                $totalReceived = $bill->receipt_payment_details->sum(function ($item) {
                    return (float) $item->rcvd_amount;
                });

                $balance = $grandAmount - $totalReceived;

                return [
                    'bill_no' => $bill->tran_no,
                    'bill_date' => \Carbon\Carbon::parse($bill->bill_date)->format('d-m-Y'),
                    'reference_no' => $bill->reference ?? '',
                    'vendor_name' => $bill->vendor->name ?? '',
                    'amount' => number_format($grandAmount, 2),
                    'balance' => number_format($balance, 2),
                ];
            });

        return response()->json(['bills' => $bills]);
    }
}